function [image_file_out, m_filetype] = vb_bias_correction_by_spm(image_file)
% Apply bias correction to image_file by spm(2/5-) function.
% [USAGE]
%    image_file_out = bias_correction_using_spm(<<image_file>>);
%
% [IN]
%        image_file : Analyze or NIfTI file (.hdr/.img)
%
% [OUT]
%    image_file_out : bias corrected filename
%                     The format is same as input file.
%                     'm' + image_file(ex. foo.hdr/img->mfoo.hdr/img)
%        m_filetype : bias corrected file type
%                     = 0 : anaylze file(.hdr/.img)
%                     = 1 : NIfTI file(.hdr/.img)
%                     = 2 : NIfTI file(.nii)
% [NOTE]
%   Before using this function, please set SPM5or8 path on your MATLAB.
%
% [SEE]
%   spm_bias_estimate.m, spm_bias_apply.m
% 
% 2010/05/12 rhayashi Initial version
% 2016/07/14 rhayashi supported NIfTI file(.nii)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Check Input file name
%
if ~exist('image_file', 'var')
    [this] = file_dialog;
    % Set extension
    this = set(this, 'file_extensions', {'.nii', '.hdr'});
    % Caption
    this.dialog_title = 'Select Image file(.nii/.hdr)';
    [image_dir, image_file] = visible(this);
    if length(image_file) == 0, return; end
    image_file = [image_dir filesep image_file{1}];
elseif exist(image_file, 'file') ~= 2
    error('Input file is not found.');
end

%
% --- Check SPM Version
%
ver_func = which('spm.m');
if isempty(ver_func)
    error('Please add SPM5/8 path to your MATLAB.');
end
func_h = @spm;
spm_ver = feval(func_h, 'Ver');
if ~(strcmpi(spm_ver, 'SPM5') || strcmpi(spm_ver, 'SPM8'))
    error('Please set SPM5/8 path');
end

%
% --- Check Original image type
%
h = msgbox('Now doing bias correction...');

% hide pushbutton
h_children = get(h, 'children');
for k=1:length(h_children)
try
    style = get(h_children(k), 'style');
    if strcmp(style, 'pushbutton')
        set(h_children(k), 'Visible', 'off');
    end
catch
end
end
drawnow; pause(0.1);

[o_ni_hdr, o_filetype] = load_nii_hdr_cbi(image_file);

[p_, f_, e_] = vb_get_file_parts(image_file);
prefix = fullfile(p_, ['m', f_]);

switch(o_filetype)
    case 0 % Analyze
        bias_corrected_hdr    = [prefix, '.hdr'];
        bias_corrected_file   = [prefix, '.img'];
        org_file = [p_, filesep, f_, '.img'];
    case 1 % NIfTI(.hdr, .img)
        bias_corrected_hdr    = [prefix, '.hdr'];
        bias_corrected_file   = [prefix, '.img'];
        org_file = [p_, filesep, f_, '.img'];
    case 2 % NIfTI(.nii)
        org_file = image_file;
        bias_corrected_hdr    = [prefix, '.nii'];
        bias_corrected_file   = [prefix, '.nii'];
    otherwise
        if ishandle(h), delete(h); end
        error('The image file type is not supported.');
end

%
% --- Bias correction
%
bias_corrected_mat    = [prefix, '.mat'];
bias_info_file        = fullfile(p_, ['bias_' f_ '.mat']);

try
    spm_bias_estimate(org_file);
catch
    if exist(bias_info_file, 'file') ~= 2
        if ishandle(h), delete(h); end
        e = lasterror;
        error(e.message);
    end
end

try
spm_bias_apply(org_file, bias_info_file);

%
% --- Check bias corrected file type
%

[m_hdr, m_filetype] = load_nii_hdr_cbi(bias_corrected_hdr);

if (o_filetype == 0) && (m_filetype == 0)
    % --- Analyze to Analyze case
    avw = avw_hdr_read(image_file);
    o_hdr = avw.hdr;

    % Change originator to original one.
    % (SPM changes originator from original to the center of the image)
    avw = avw_hdr_read(bias_corrected_file);
    avw.hdr.hist.originator = o_hdr.hist.originator;
    avw.hdr.hist.orient     = o_hdr.hist.orient;
    avw_hdr_write(avw);
elseif (o_filetype == 0) && (m_filetype == 1)
    % --- Analyze to NIfTI case
    %  SPM8(or greater) changes analyze to NIfTI format automatically.
    %  So this program changes header from NIfTI image to Analyze.

    % Change originator to original one.
    % (SPM changes originator from original to the center of the image)
    avw = avw_hdr_read(image_file);
    o_hdr = avw.hdr;

    avw = avw_hdr_read(bias_corrected_file);
    avw.hdr.hist.originator = o_hdr.hist.originator;
    avw.hdr.hist.orient     = o_hdr.hist.orient;
    if isempty(o_hdr.hist.smin)
        o_hdr.hist.smin = 0;
    end
    avw.hdr.hist.smin       = o_hdr.hist.smin;
    avw_hdr_write(avw);

elseif (o_filetype == 1) && (m_filetype == 1)
    % --- NIfTI to NIfTI case

    % There is nothing to do.
elseif (o_filetype == 2) && (m_filetype == 2)
    % --- NIfTI(.nii) to NIfTI case(.nii)
    prefix = [prefix, '.nii'];
elseif (o_filetype == 1) && (m_filetype == 0)
    % --- NIfTI to Analyze case
    if ishandle(h), delete(h); end
    error('NIfTI file is processed by SPM2. This might occur by program bug.');
end

% Remove bias correction info file
if exist(bias_info_file, 'file') == 2
    delete(bias_info_file);
end

% Remove SPM specific file
if exist(bias_corrected_mat, 'file') == 2
    delete(bias_corrected_mat);
end

% Output filename
image_file_out = prefix;

disp('Bias correction is finished.');
if ishandle(h), delete(h); end

catch
if ishandle(h), delete(h); end
e = lasterror;
error(e.message);
end
