function megmat_file = neuromag_to_megmatx(data_fiffile, mri_niifile, ...
  mri_fidsfile, headpoints_file, output_prefix, face_file, sensor_mode)
% make MEG-MAT & EEG-MAT files from neuromag fiffile.
% [USAGE]
%    neuromag_to_megmatx(data_fiffile, mri_niifile, ...
%                       output_prefix, face_file, sensor_mode)
% [IN]
%      data_fiffile : Data fiffile.
%       mri_niifile : MRI niifile.
%      mri_fidsfile : MAT file containing MRI coordinate values of LPA,NAS,RPA.
%   headpoints_file : Headpoints in MRI coordinate
%     output_prefix : output filename prefix.
%                     (ex. '/home/RH/RH' => /home/RH/RH.meg.mat)
% [optional input]
%         face_file : If you want to calculate by using spherical model,
%                     center position and radius of spherical model are 
%                     necessary.
%                     A face file which is made by positioning tool has them.
%                     For face file, please refer to the manual of the
%                     positioning tool
%       sensor_mode : if sensor_mode = 'Planar', planar sensor is selected
%                     otherwise all MEG sensor is selected
% [OUT]
%    meg.mat & eeg.mat files with output_prefix are saved
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)
%
% 2012/11/16 M.Fukushima
% * modified to import Henson dataset
% 2013/02/09 M.Fukushima
% * added headpoints import

%
% --- Previous check
%
if ~exist('data_fiffile', 'var')
    error('data_fiffile is a required parameter.');
end
if ~exist('mri_niifile', 'var')
    error('mri_niifile is a required parameter.');
end
if ~exist('mri_fidsfile', 'var')
    error('mri_fidsfile is a required parameter.');
end
if ~exist('output_prefix', 'var')
    error('output_prefix is a required parameter.');
end
if ~exist('face_file', 'var')
    face_file = [];
end
if exist(data_fiffile, 'file') ~= 2
    error('data_fiffile:%s is not found.', data_fiffile);
end
if exist(mri_niifile, 'file') ~= 2
    error('mri_niifile:%s is not found.', mri_niifile);
end
if exist(mri_fidsfile, 'file') ~= 2
    error('mri_fidsfile:%s is not found', mri_fidsfile);
end
if exist(headpoints_file, 'file') ~= 2
    error('headpoints_file:%s is not found', headpoints_file);
end
%if ~exist('analyzefile', 'var')
%    error('analyzefile is a required parameter.');
%end
%if exist(analyzefile, 'file') ~= 2
%    error('analyzefile:%s is not found.', analyzefile);
%end
[f_path] = vb_get_file_parts(output_prefix);
if exist(f_path, 'dir') ~= 7
    error('output_prefix directory:%s is invalid.', f_path);
end
if ~exist('sensor_mode','var'), sensor_mode = 'All'; end

%
% --- Main Procedure
%


%
% --- Load Header
%
fprintf('--- now loading neuromag header.\n');
[MEGinfo_list, sensor_info] = neuromag_load_meg_headerx(data_fiffile, ...
                                mri_niifile, mri_fidsfile, headpoints_file);%, ...
%                                                       analyzefile);

%
% --- Make sphere model
%
if ~isempty(face_file)

	if exist(face_file,'file') ~= 2
    warning('face_file:%s is not found.\n', face_file);
%     error('face_file:%s is not found.', face_file);
  end
  
  [state, guide_def] = ...
    vb_util_check_variable_in_matfile(face_file, 'spherical_head');

  if state == guide_def.VALID
    load(face_file, 'spherical_head');
    Vcenter = spherical_head.Vcenter;
    Vradius = spherical_head.Vradius;
  else
    [state, guide_def] = ...
      vb_util_check_variable_in_matfile(face_file, 'surf_face');
    if state == guide_def.VALID
      % --- calculate center position and radius of spherical model
      % --- Upper face is used for fitting sphere model 
      %	--- ix = find(V(:,3) > (h * Zmax + (1-h) * Zmin));
      h = 0.4;
    	load(face_file, 'surf_face')
	
    	V = surf_face.V_reduce;
    	Zmax = max(V(:,3));
    	Zmin = min(V(:,3));
    	ix = find(V(:,3) > (h * Zmax + (1-h) * Zmin));
	
    	% Fit face to sphere 
    	[Vcenter,Vradius] = vb_center_sphere(V(ix,:));
    else
    	Vcenter = [];
    	Vradius = [];
    end
  end
    
else
	Vcenter = [];
	Vradius = [];
end

pick   = sensor_info.pick;
Qpick  = sensor_info.Qpick;
weight = sensor_info.Weight;

if strcmp(sensor_mode,'Planar'),
	% select planar sensor from MEG data
    MEGinfo_base = MEGinfo_list{1};
	ix_ch = find( MEGinfo_base.ChannelInfo.Type == 3 );
	
	weight  = weight(ix_ch,:);
	ix_coil = find( sum(abs(weight),1) > 0);
	weight  = weight(:,ix_coil);
	
	pick  = pick(ix_coil,:);
	Qpick = Qpick(ix_coil,:);
else
	ix_ch = [];
end

Ninfo = length(MEGinfo_list);

%
% --- Make MEG-MAT & EEG-MAT files.
%
fprintf('--- now loading neuromag data.\n');
for k=1:Ninfo
  % -- Make file name
  if Ninfo == 1
    megmat_file{1} = [output_prefix, '.meg.mat'];
    eegmat_file{1} = [output_prefix, '.eeg.mat'];
  else
    megmat_file{k} = [output_prefix, '_', num2str(k), '.meg.mat'];
    eegmat_file{k} = [output_prefix, '_', num2str(k), '.eeg.mat'];
  end

  % load data 
  [bexp, ch_info, sfreq] = neuromag_load_meg_data(data_fiffile);
  [bexp_ext, extra_info] = neuromag_load_extra_data(data_fiffile);


  % Set MEGinfo
  MEGinfo = MEGinfo_list{k};
      
  if ~isempty(Vcenter)
	  MEGinfo.Vcenter = Vcenter;
	  MEGinfo.Vradius = Vradius;
  end
	
  MEGinfo.Nsample    = size(bexp, 2);
  MEGinfo.SampleFreq = sfreq;

  % ----- MEGinfo.Trial
  Trial(1).number = 1;
  Trial(1).sample = 1:MEGinfo.Nsample;
  Trial(1).Active = true;
  MEGinfo.Trial = Trial;

  MEGinfo.sensor_weight = weight;
  MEGinfo.ExtraChannelInfo = extra_info;

  MEGinfo.device   = 'NEUROMAG';
  CoordType   = 'SPM_Right_m';
    
  Measurement = 'MEG';

  if ~isempty(ix_ch)
  	bexp = bexp(ix_ch,:);
  end
    
  
  MEGinfo.Measurement = Measurement;
  MEGinfo.ActiveChannel = MEGinfo.ChannelInfo.Active;
  MEGinfo.ActiveTrial = MEGinfo.Trial.Active;
  
  % save MEG-MAT file
  vb_fsave(megmat_file{k}, 'MEGinfo', 'Measurement', 'CoordType', ...
  	 'pick', 'Qpick', 'bexp', 'bexp_ext');
  fprintf('%s was created.\n', megmat_file{k});
  
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
% RHAYASHI output EEG-MAT not supported yet.
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
%   % --- Make EEG-MAT file
%   ind_eeg = [1:60 65:74];
%   eeg_data = bexp_ext(ind_eeg,:);
%   Measurement = 'EEG';
%   EEGinfo.Measurement = Measurement;
%   EEGinfo.Device = MEGinfo.device;
%   EEGinfo.Nchannel = 70;
%   EEGinfo.Nsample = MEGinfo.Nsample;
%   EEGinfo.Nrepeat = MEGinfo.Nrepeat;
%   EEGinfo.Pretrigger = MEGinfo.Pretrigger;
%   EEGinfo.SampleFrequency = MEGinfo.SampleFreq;
%   EEGinfo.ChannelID = (1:EEGinfo.Nchannel)';
%   EEGinfo.ChannelName = MEGinfo.ExtraChannelInfo.Channel_name(ind_eeg);
%   EEGinfo.ActiveChannel = ones(EEGinfo.Nchannel,1);
%   EEGinfo.ActiveTrial = ones(EEGinfo.Nrepeat,1);
%   EEGinfo.Vcenter = MEGinfo.Vcenter;
%   EEGinfo.Vradius = MEGinfo.Vradius;
%   EEGinfo.MRI_ID = MEGinfo.MRI_ID;
%   EEGinfo.Trial = MEGinfo.Trial;
%   EEGinfo.ChannelInfo.ID = EEGinfo.ChannelID;
%   EEGinfo.ChannelInfo.Name = EEGinfo.ChannelName;
%   EEGinfo.ChannelInfo.Type = ones(EEGinfo.Nchannel,1);
%   EEGinfo.ChannelInfo.Active = EEGinfo.ActiveChannel;
%   EEGinfo.ChannelInfo.PhysicalUnit = cell(EEGinfo.Nchannel,1);
%   for ii = 1:EEGinfo.Nchannel
%     EEGinfo.ChannelInfo.PhysicalUnit{ii} = 'V';
%   end
%   EEGinfo.ExtraChannelInfo.Channel_id = [];
%   EEGinfo.ExtraChannelInfo.Channel_name = {};
%   EEGinfo.ExtraChannelInfo.Channel_type = [];
%   EEGinfo.ExtraChannelInfo.Channel_active = [];
%   EEGinfo.ExtraChannelInfo.PhysicalUnit = {};
%   for ii = 1:(EEGinfo.Nchannel + length(EEGinfo.ExtraChannelInfo.Channel_id))
%     EEGinfo.DataType{ii} = 'Unknown';
%   end
%   
%   [R, KI, NU] = hpipoints(data_fiffile);
%   CoordHead = R(:,KI==3)';
%   [ans, ind_tmp] = intersect(NU(KI==3),ind_eeg);
%   CoordHead = CoordHead(ind_tmp,:);
%   % load voxel data and transform matrix from mri niifile.
%   [voxel, mri_trans_info] = neuromag_load_mri_infox...
%       (data_fiffile, mri_niifile, mri_fidsfile, headpoints_file);
%   Vdim = size(voxel);
%   % transform coil coordinates in head coordinate to SPM coordinate.
%   Coord  = neuro_head_to_spm_right(CoordHead, mri_trans_info, Vdim);
%   
%   EEGinfo.Coord = Coord;
%   EEGinfo.CoordType = CoordType;
%   
%   % save EEG-MAT file
%   vb_fsave(eegmat_file{k}, 'EEGinfo', 'Measurement', 'eeg_data');
%   fprintf('%s was created.\n', eegmat_file{k});
end
fprintf('--- conversion were finished.\n');
