function [MEGinfo_list, sensor_info] = ...
            neuromag_load_meg_header(data_fiffile, mri_fiffile)
% load sensor and data information from fiffile.
% [USAGE]
%   [MEGinfo_list, sensor_info] = ...
%     neuromag_load_meg_header(data_fiffile, mri_fiffile);
% [IN]
%    data_fiffile : data fiffile.
%     mri_fiffile : MRI fiffile.
% [OUT]
%    MEGinfo_list : <<struct array>> array of MEGinfo [1 x N_fileset]
%     sensor_info : <<struct>> sensor information
%            .pick       : n-th coil position  (Right-hand SPM coordinate) [m]
%                        :  [Ncoil x 3]
%            .Qpick      : n-th coil direction (Right-hand SPM coordinate)
%                        :  [Ncoil x 3]
%            .coord_type : coordinate system of pick, Qpick.
%            .Weight     : n-th coil weight for m-th channel [Nchannel x Ncoil]
%                        : basis(channel,dipole) = 
%                            CoilWeight * basis(coil,dipole)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('data_fiffile', 'var')
    error('data_fiffile is a required parameter.');
end
if ~exist('mri_fiffile', 'var')
    error('mri_fiffile is a required parameter.');
end
if exist(data_fiffile, 'file') ~= 2
    error('data_fiffile:%s is not found.', data_fiffile);
end
if exist(mri_fiffile, 'file') ~= 2
    error('mri_fiffile:%s is not found', mri_fiffile);
end

%if ~exist('analyzefile', 'var')
%    error('analyzefile is a required parameter.');
%end
%if exist(analyzefile, 'file') ~= 2
%    error('analyzefile:%s is not found', analyzefile);
%end


%
% --- Extract information
%
%[B, Vdim, Vsize] = vb_load_analyze_to_right(analyzefile);

% load channel information and coordinate transform matrix
% from the timeseries fiffile.
[ch_info, pos_info] = neuromag_load_ch_info(data_fiffile);

% load voxel data and transform matrix from mri fiffile.
[voxel, mri_trans_info] = neuromag_load_mri_info(mri_fiffile);

% transform coil coordinates from coil coordinate system to head coordinate system.
[pick_head, Qpick_head, Weight] = get_pos_neuromag(ch_info);

Vdim = size(voxel);

% transform coil coordinates in head coordinate to SPM coordinate.
pick  = neuro_head_to_spm_right(pick_head, mri_trans_info, Vdim);
% only rotation
Qpick = neuro_head_to_spm_right(Qpick_head, mri_trans_info, Vdim, 1);


% Check the number of data sets in the fiffile.
info = fiff_find_evoked(data_fiffile);

if isempty(info)
    Nsets = 0;
else
    Nsets = length(info);
end

if Nsets == 0
    % Continuous data
    trial_header_list = cell(1);

    % read header
    header = struct;
    header.Nrepeat    = 1;
    header.Pretrigger = 0;
    header.Comment    = '';
    header.acq_type   = 'Continuous_Raw';
    header.index_in_fiffile = -1;

    trial_header_list{1} = header;
    Nsets = 1;

else
   % Averaged data
    data_sets = fiff_read_evoked_all(data_fiffile);
    trial_header_list = cell(Nsets, 1);
    for k=1:Nsets
        info_k = data_sets.evoked(k);
        header = struct;
        header.Nsample    = size(info_k.epochs, 2);
        header.Nrepeat    = size(info_k.epochs, 3);
        header.SampleFreq = data_sets.info.sfreq;
        header.Pretrigger = find(info_k.times == 0)-1;
        header.Comment    = info_k.comment;
        header.acq_type   = 'Evoked_Ave';
        header.index_in_fiffile = k;
        trial_header_list{k} = header;
   end
end

%
% --- Make information
%
d = vb_define_coordinate;

% Make Sensor infomation
sensor_info = struct;
sensor_info.pick  = pick;
sensor_info.Qpick = Qpick;
sensor_info.coord_type = d.COORDINATE_SPM_RIGHT_M;
sensor_info.Weight = Weight;

% --- Sensor_type info
NEUROMAG_MagnetoMeter = ... % VBMEG's magnetometer id
  neuromag_get_channel_type('NEUROMAG_MagnetoMeter');
NEUROMAG_PlanarGradioMeter = ... % VBMEG's planarmeter id
  neuromag_get_channel_type('NEUROMAG_PlanarGradioMeter');

% --- sensor basic information
n_ch = size(ch_info.channel_name, 1);
sensor_type = zeros(n_ch, 1);
sensor_name = ch_info.channel_ix;
sensor_id   = 1:n_ch;

% ----- Convert channel type from neuromag to vbmeg's(MagnetoMeter)
ix = ch_info.sensor_type == 0;
sensor_type(ix) = NEUROMAG_MagnetoMeter;

% ----- Convert channel type from neuromag to vbmeg's(PlanarGradioMeter)
ix = ch_info.sensor_type == 1;
sensor_type(ix) = NEUROMAG_PlanarGradioMeter;

[f_path, f_name] = vb_get_file_parts(data_fiffile);

% -- Make MEGinfo
MEGinfo_list = cell(Nsets, 1);
for k=1:Nsets
  MEGinfo = struct;
  % -- MEGinfo device properties
  MEGinfo.MEG_ID   = f_name;
    
  % -- MEGinfo data properties
  MEGinfo.Nchannel      = ch_info.Nch;
%  MEGinfo.Nsample       = trial_header_list{k}.Nsample;
  MEGinfo.Pretrigger    = trial_header_list{k}.Pretrigger;
%  MEGinfo.SampleFreq    = trial_header_list{k}.SampleFreq;
  MEGinfo.sensor_weight = sensor_info.Weight;
    
  MEGinfo.MEGch_id   = sensor_id'; % [n_channel x 1]
  MEGinfo.MEGch_name = sensor_name;

  % ----- MEGinfo .ChannelInfo
  ChannelInfo.Active = ones(n_ch, 1);
  ChannelInfo.ID     = MEGinfo.MEGch_id;
  ChannelInfo.Name   = MEGinfo.MEGch_name;
  ChannelInfo.Type   = sensor_type;
  MEGinfo.ChannelInfo= ChannelInfo;

  MEGinfo.MRI_ID        = '';
  MEGinfo.Nrepeat     = trial_header_list{k}.Nrepeat;

  % -- MEGinfo sphere model
  MEGinfo.Vcenter = [];
  MEGinfo.Vradius = [];

  % ----- device dependent information
  device_info.acq_type = trial_header_list{k}.acq_type;

  % ------- Neuromag specific information
  device_info.index_in_fiffile = trial_header_list{k}.index_in_fiffile; 
  device_info.Comment = trial_header_list{k}.Comment;

  MEGinfo.device_info = device_info;

  MEGinfo_list{k} = MEGinfo;
end

return;

% --- END OF FILE --- %
