function [lh_clutfile, rh_clutfile, Nlabel_l, Nlabel_r] = ...
                     dmri_color_table_create_for_parcels(parcel_dir, output_dir)
% save color tabels
%
% [Usage]
%    [lh_clutfile, rh_clutfile,  Nlabel_l, Nlabel_r] = ...
%               dmri_color_table_create_for_parcels(parcel_dir, output_dir);
%
% [Input]
%    parcel_dir : parcellation directory that contains label directory.
%                 (parcel_dir/parcels_label)
%    output_dir : output directory of color table files.
%                 = output_dir/lh.clut.txt : left hemisphere color table file.
%                 = output_dir/rh.clut.txt : right hemisphere color table file.
% [Output]
%    lh_clutfile : left hemisphere color table file.
%    rh_clutfile : right hemisphere color table file.
%       Nlabel_l : The number of left label.
%       Nlabel_r : The number of right label.
%
% 2012/06/24 M.Fukushima
% 2012/12/02 M.Fukushima
% 2014/06/09 M.Fukushima
% 2014/06/18 rhayashi
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 2
    error('Please check input arguments.');
end
if exist(parcel_dir, 'dir') ~= 7
    error('Specified parcel_dir not found.');
end
if exist(output_dir, 'dir') ~= 7
    error('Specified output_dir not found.');
end

%
% --- Main Procedure
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Get parcel information filename
parcel_label1 = fullfile(parcel_dir, filesep, ...
                        'parcels_label', filesep, ...
                        'parcel1.label');

if exist(parcel_label1, 'file') ~= 2
    error('Parcel label file not found in [parcel_dir/parcels_label].');
end
fid = fopen(parcel_label1, 'rt');
if fid == -1
    error('Cannot open parcel label file.');
end
% read first line
first_line = fgetl(fid);
fclose(fid);

% extract parcel file name.
first_line  = strrep(first_line, '#!ascii label, from subject ', '');
parcel_file = first_line(1:end-1);


%
% --- Create Color table files
%
load(parcel_file);
Nlabel_l = length(lh_cortex_id);
Nlabel_r = length(rh_cortex_id);

if Nlabel_l >= 1000
    disp(sprintf('Parcellation areas on Left hemisphere %d(MAX:999). Skipped to show lh_cortex_id(1000~%d).', Nlabel_l, Nlabel_l));
    Nlabel_l = 999;
end
if Nlabel_r >= 1000
    disp(sprintf('Parcellation areas on Right hemisphere %d(MAX:999). Skipped to show rh_cortex_id(1000~%d).', Nlabel_r, Nlabel_r));
    Nlabel_r = 999;
end

% lindex
lh_lindex = (1:Nlabel_l)';
rh_lindex = ([1:Nlabel_r]+Nlabel_l)';

% lrgb
base = 255:-20:0;
Nb = length(base);

lrgb = zeros(Nb^3,3);
lrgb(:,1) = repmat(base,[1 Nb^2])';
lrgb(:,2) = reshape(repmat(base,[Nb Nb]),[1 Nb^3])';
lrgb(:,3) = reshape(repmat(repmat(base,[Nb 1]),[Nb 1]),[1 Nb^3])';

% % Random
% lrgb = round(255*rand(size(lrgb)));

lh_lrgb = lrgb(1:Nlabel_l,:);
rh_lrgb = lrgb((Nlabel_l+1):(Nlabel_l+Nlabel_r),:);

% lvals
lh_lvals = zeros(Nlabel_l,1);
rh_lvals = zeros(Nlabel_r,1);

% clutfile
lh_clutfile = fullfile(output_dir, filesep, 'lh.clut.txt');
rh_clutfile = fullfile(output_dir, filesep, 'rh.clut.txt');

% Write .label file
% lh
ok = write_clut(lh_lindex, lh_lrgb, lh_lvals, lh_clutfile);
if ok ~=1
  error('Writing .label file is failed!!');
end
% rh
ok = write_clut(rh_lindex, rh_lrgb, rh_lvals, rh_clutfile);
if ok ~=1
  error('Writing .label file is failed!!');
end

