function dmri_wm_FA_volume_create(fs_dir, ...
                                  ref_struct_niigz_file, ...
                                  ref_FA_niigz_file, trans_info_dir, ...
                                  output_wm_niigz_file)
% Create whitematter volume on FA coordinate system.
% The whitematter is the one created by FreeSurfer(lh.smoothwm, rh/smoothwm).
%
% [Usage]
%    dmri_wm_FA_volume_create(fs_dir, ...
%                             ref_struct_niigz_file, ...
%                             ref_FA_niigz_file, trans_info_dir, ...
%                             output_wm_niigz_file)
%
% [Input]
%                   fs_dir : Freesurfer subject directory.
%    ref_struct_niigz_file : reference struct brain image file (niftigz format).
%        ref_FA_niigz_file : reference FA(Fractional anisotrophy) image file (niftigz format).
%           trans_info_dir : The directory which cnotains transform infomation files
%                            created by dti_transwarp_info_create.m
%     output_wm_niigz_file : White matter volume data on FA coordinate system.
%
% [Output]
%    none
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 5
    error('Please check input arguments.');
end

if exist(fs_dir, 'dir') ~= 7
    error('Specified fs_dir not found.');
end
if exist(ref_struct_niigz_file, 'file') ~= 2
    error('Specified ref_struct_niigz_file not found.');
end
if exist(ref_FA_niigz_file, 'file') ~= 2
    error('Specified ref_FA_niigz_file not found.');
end
if exist(trans_info_dir, 'dir') ~= 7
    error('Specified trans_info_dir not found.');
end

%
% --- Main Procedure
%
disp('Creating whitematter volume on FA coordinate system.');
if exist(output_wm_niigz_file, 'file') == 2
    copyfile(output_wm_niigz_file, [output_wm_niigz_file, '.bak']);
    delete(output_wm_niigz_file);
end
% Create working directory
work_dir = tempname;
mkdir(work_dir);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% read white matter label
%
lh_asc_file = fullfile(fs_dir, filesep, '/bem/lh.smoothwm.asc');
rh_asc_file = fullfile(fs_dir, filesep, '/bem/rh.smoothwm.asc');

[l_S, l_v, l_f] = read_asc(lh_asc_file);
[r_S, r_v, r_f] = read_asc(rh_asc_file);

% Create .label file of White matter
lindex = 0:(size(l_v,1) + size(r_v,1) - 1);
lxyz = [l_v(:,1:3)' r_v(:,1:3)']';
lvals = [l_v(:,4)' r_v(:,4)']';

wm_label_file = fullfile(work_dir, filesep, 'wm.label');
ok = write_label(lindex, lxyz, lvals, wm_label_file, fs_dir);
if ok ~=1
  error('Writing .label file is failed!!');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create white matter volume
%
lh_wm_file = fullfile(fs_dir, filesep, 'surf/lh.smoothwm');
rh_wm_file = fullfile(fs_dir, filesep, 'surf/rh.smoothwm');

tmp_wm_lh_fill_file = fullfile(work_dir, filesep, 'tmp_wm_lhfill.nii.gz');
tmp_wm_rh_fill_file = fullfile(work_dir, filesep, 'tmp_wm_rhfill.nii.gz');

% Fill inner white matter surface
cmd = ['mris_fill', ...
       ' -c -r 1', ...
       ' ', lh_wm_file, ...
       ' ', tmp_wm_lh_fill_file];

dmri_system(cmd, '-echo');

cmd = ['mris_fill', ...
       ' -c -r 1', ...
       ' ', rh_wm_file, ...
       ' ', tmp_wm_rh_fill_file];
dmri_system(cmd, '-echo');

% lh_fill + rh_fill = fill
tmp_wm_fill_file = fullfile(work_dir, filesep, 'tmp_wm_fill.nii.gz');
dmri_add_images(tmp_wm_lh_fill_file, ...
                tmp_wm_rh_fill_file, ...
                tmp_wm_fill_file);

% Convert label to volume(only label)
brain_on_FS_coord_file = fullfile(fs_dir, filesep, 'mri/brain.nii.gz');
tmp_wm_file            = fullfile(work_dir, filesep, 'tmp_wm.nii.gz');

cmd = ['mri_label2vol', ...
       ' --label ', wm_label_file, ...
       ' --temp ' , brain_on_FS_coord_file, ... % Template. the same size and geometry as the template.
       ' --o '    , tmp_wm_file, ... % Single frame output volume.
       ' --identity'];
dmri_system(cmd, '-echo');

% Create whitematter with label
%  tmp_wm_sum.nii.gz  = tmp_wm_fill_file(whitematter) + tmp_wm.nii.gz(label)
tmp_wm_sum_file = fullfile(work_dir, filesep, 'tmp_wm_sum.nii.gz');
dmri_add_images(tmp_wm_fill_file, ...
                tmp_wm_file, ...
                tmp_wm_sum_file);

% Binarize
tmp_wm_sum_bin_file = fullfile(work_dir, filesep, 'tmp_wm_sum_bin.nii.gz');
cmd = ['fslmaths', ...
       ' ', tmp_wm_sum_file, ...
       ' -thr 0 ', ...
       ' -bin '  , tmp_wm_sum_bin_file];
dmri_system(cmd);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Convert from FreeSurfer space to FA space

% Convert from FreeSurfer to Stuct image space
tmp_wm_sum_bin_on_struct_coord_file = fullfile(work_dir, filesep, 'tmp_wm_sum_bin_on_structcoord.nii.gz');
dmri_image_transform(tmp_wm_sum_bin_file, ...
                    tmp_wm_sum_bin_on_struct_coord_file, ...
                    ref_struct_niigz_file, ...
                    'fs2struct', ...
                    trans_info_dir);

% Convert Struct image space to FA space
output_dir = fileparts(output_wm_niigz_file);
if exist(output_dir, 'dir') ~= 7
    mkdir(output_dir);
end
dmri_image_transform(tmp_wm_sum_bin_on_struct_coord_file, ...
                    output_wm_niigz_file, ...
                    ref_FA_niigz_file, ...
                    'struct2fa_nointerp', ...
                    trans_info_dir);
fprintf('White matter volume file created : %s\n', output_wm_niigz_file);

% delete working directory
rmdir(work_dir, 's');
