function dmri_cortex_parcel_from_area(brain_file, fs_info_file, parcel_area_file, parcel_file)
% Get indices of parcellated brain
% [Usage]
%    dmri_cortex_parcel_from_area(brain_file, fs_info_file, parcel_area_file, parcel_file);
%
% [Input]
%       brain_file : VBMEG BRAIN-MAT file(.brain.mat)
%     fs_info_file : Freesurfer information file.
% parcel_area_file : Parcellation is done with the area file(.area.mat).
%      parcel_file : Parcel file.
%
% Original code is vb_original_normal_statics.m
% 2012/12/02 M.Fukushima
% 2017/10/25 Y.Takeda
%  Modified for parceling cortex according to area file
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 4
    error('Please check input arguments.');
end
if exist(brain_file, 'file') ~= 2
    error('brain_file not found.');
end
if exist(fs_info_file, 'file') ~= 2
    error('fs_info_file not found.');
end

%
% --- Main Procedure
%
disp('Cortex parcellation..');
start = tic;

% Max radius of neighbor search
Rmax = 0.004;	% 4mm

%
% --- Load original brain
%
load(fs_info_file);

brain_parm.FS_left_file  = FS_wm.lh_smoothwm_asc_file;
brain_parm.FS_right_file = FS_wm.rh_smoothwm_asc_file;

vb_disp('Load original brain')
[V0L,F0L,n0L,V0R,F0R,n0R] = vb_load_orig_brain_surf(brain_parm);
%
% --- SPM cordinate in [m] , Normal vector is outward.
%
vb_disp('Change coordinate to [m] ');

V0L = V0L/1000;
V0R = V0R/1000;

%
% --- Set normal direction outward
%     normal vector for disconnected vertex is set to zero
tic
vb_disp_nonl('Set normal direction outward: ')
[F0L, V0L, xxL] = vb_out_normal_surf(F0L,V0L);

xxsig = sign( sum( n0L .* xxL , 2) );
n0L = n0L .* repmat(xxsig, [1 3]);

[F0R, V0R, xxR] = vb_out_normal_surf(F0R,V0R);

xxsig = sign( sum( n0R .* xxR , 2) );
n0R = n0R .* repmat(xxsig, [1 3]);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

% Original vertex index for cortex
load(brain_file, 'subj');
[tmp1, tmp2, BV_index] = vb_load_cortex_info(brain_file, 'subj');

NV0L  = size(V0L,1);
indxL = BV_index.Left;
indxR = BV_index.Right;

if min(indxR) > NV0L,
	indxR = indxR - NV0L;
end

%
% Search neighbor points along cortex sheet 
% and calculate normal direction statistics
%

%
% --- Make neighbor member list of original brain (Left)
%
tic
vb_disp_nonl('Find neighbor index (Left): ')
[xxDL,xxFL]   = vb_next_distance( F0L, V0L ); 
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

vb_disp_nonl('Make neighbor member list of original brain (Left): ')
tic
Llist  = vb_find_near_member(xxFL, xxDL, indxL, Rmax);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

%
% --- Make neighbor member list of original brain (Right)
%
tic
vb_disp_nonl('Find neighbor index (Right): ')
[xxDR,xxFR]   = vb_next_distance( F0R, V0R ); 
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

vb_disp_nonl('Make neighbor member list of original brain (Right): ')
tic
Rlist  = vb_find_near_member(xxFR, xxDR, indxR, Rmax);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

%% Whole Brain Parcellation
% Search radious for parcellation (as large as possible)
Rmax = 0.040; % 4cm

% Parcel cortex according to all the vertices in brain model
vbmeg_area_ix=1:length(indxL)+length(indxR);

Nroot_l = length(indxL);
tmp = find(vbmeg_area_ix<=Nroot_l);
half = tmp(end);

% Parcellation
[lh_id1] = vb_find_near_member...
  (xxFL, xxDL, indxL(vbmeg_area_ix(1:half)), Rmax);
[rh_id1] = vb_find_near_member...
  (xxFR, xxDR, indxR(vbmeg_area_ix(half+1:end)-Nroot_l), Rmax);

% Combine parcels according to area file
load(parcel_area_file,'Area')
Narea=length(Area);
al=1;
ar=1;
for area=1:Narea
    ix=Area{area}.Iextract;
    ix1=[];
    if ix(1)<=Nroot_l% Left
        for i=1:length(ix)
            ix1=[ix1,lh_id1{ix(i)}];
        end
        lh_id2{al}=sort(ix1);
        area_num_l1(al)=area;
        al=al+1;
    else% Right
        ix=ix-Nroot_l;
        for i=1:length(ix)
            ix1=[ix1,rh_id1{ix(i)}];
        end
        rh_id2{ar}=sort(ix1);
        area_num_r1(ar)=area;
        ar=ar+1;
    end
end

% Make (lh/rh)_cortex_id1 which contains only cortex index (remove subcortex elements)
lh_cortex_id1 = cell(length(lh_id2), 1);
rh_cortex_id1 = cell(length(rh_id2), 1);
for n=1:length(lh_id2)
    lh_cortex_id1{n} = lh_id2{n};
    [tmp, ind] = intersect(lh_id2{n}, FS_wm.lh_subcortex_index);
    lh_cortex_id1{n}(ind) = [];
end
for n=1:length(rh_id2)
    rh_cortex_id1{n} = rh_id2{n};
    [tmp, ind] = intersect(rh_id2{n}, FS_wm.rh_subcortex_index);
    rh_cortex_id1{n}(ind) = [];
end

% Extract parcels which have no or small subcortex
th_p=0.1;% Threshold for portion of subcortex
p_subcortex=zeros(length(lh_cortex_id1),1);
for n=1:length(lh_cortex_id1)
    p_subcortex(n)=(length(lh_id2{n})-length(lh_cortex_id1{n}))/length(lh_id2{n});
end
ix=find(p_subcortex<th_p);
lh_id=lh_id2(ix);
lh_cortex_id=lh_cortex_id1(ix);
area_num_l=area_num_l1(ix);
p_subcortex=zeros(length(rh_cortex_id1),1);
for n=1:length(rh_cortex_id1)
    p_subcortex(n)=(length(rh_id2{n})-length(rh_cortex_id1{n}))/length(rh_id2{n});
end
ix=find(p_subcortex<th_p);
rh_id=rh_id2(ix);
rh_cortex_id=rh_cortex_id1(ix);
area_num_r=area_num_r1(ix);

% Make vbmeg_area_ix
Area_cortex=Area([area_num_l,area_num_r]);
Narea=length(Area_cortex);
V=vb_load_cortex(brain_file,'Inflate');
vbmeg_area_ix=zeros(Narea,1);
for area=1:Narea
    ix=Area_cortex{area}.Iextract;
    center=mean(V(ix,:),1);
    d=sum((V(ix,:)-repmat(center,length(ix),1)).^2,2);
    [~,b]=min(d);
    vbmeg_area_ix(area)=ix(b);
end

% Save index
save(parcel_file, 'Area_cortex', 'vbmeg_area_ix', 'lh_id', 'rh_id', 'lh_cortex_id', 'rh_cortex_id', 'brain_file');

% Create membership matrix(parcels to brain model(V))
[membershipmat] = make_membershipmat(brain_file, parcel_file);
save(parcel_file, '-append', 'membershipmat');

disp(sprintf('\nSaved:%s', parcel_file));

toc(start);
