function dmri_cortex_parcel(brain_file, fs_info_file, num_vertex, parcel_file)
% Get indices of parcellated brain
% [Usage]
%    dmri_cortex_parcel(brain_file, fs_info_file, num_vertex, parcel_file);
%
% [Input]
%      brain_file : VBMEG BRAIN-MAT file(.brain.mat)
%    fs_info_file : Freesurfer information file.
%      num_vertex : The number of verticies.
%     parcel_file : Parcel file.
%
% Original code is vb_original_normal_statics.m
% 2012/12/02 M.Fukushima
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if nargin ~= 4
    error('Please check input arguments.');
end
if exist(brain_file, 'file') ~= 2
    error('brain_file not found.');
end
if exist(fs_info_file, 'file') ~= 2
    error('fs_info_file not found.');
end

%
% --- Main Procedure
%
disp('Cortex parcellation..');
start = tic;
dmri_vbmeg_vertex_select(brain_file, fs_info_file, num_vertex, parcel_file);

% Max radius of neighbor search
Rmax = 0.004;	% 4mm

%
% --- Load original brain
%
load(fs_info_file);

brain_parm.FS_left_file  = FS_wm.lh_smoothwm_asc_file;
brain_parm.FS_right_file = FS_wm.rh_smoothwm_asc_file;

vb_disp('Load original brain')
[V0L,F0L,n0L,V0R,F0R,n0R] = vb_load_orig_brain_surf(brain_parm);
%
% --- SPM cordinate in [m] , Normal vector is outward.
%
vb_disp('Change coordinate to [m] ');

V0L = V0L/1000;
V0R = V0R/1000;

%
% --- Set normal direction outward
%     normal vector for disconnected vertex is set to zero
tic
vb_disp_nonl('Set normal direction outward: ')
[F0L, V0L, xxL] = vb_out_normal_surf(F0L,V0L);

xxsig = sign( sum( n0L .* xxL , 2) );
n0L = n0L .* repmat(xxsig, [1 3]);

[F0R, V0R, xxR] = vb_out_normal_surf(F0R,V0R);

xxsig = sign( sum( n0R .* xxR , 2) );
n0R = n0R .* repmat(xxsig, [1 3]);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

% Original vertex index for cortex
load(brain_file, 'subj');
[tmp1, tmp2, BV_index] = vb_load_cortex_info(brain_file, 'subj');

NV0L  = size(V0L,1);
indxL = BV_index.Left;
indxR = BV_index.Right;

if min(indxR) > NV0L,
	indxR = indxR - NV0L;
end

%
% Search neighbor points along cortex sheet 
% and calculate normal direction statistics
%

%
% --- Make neighbor member list of original brain (Left)
%
tic
vb_disp_nonl('Find neighbor index (Left): ')
[xxDL,xxFL]   = vb_next_distance( F0L, V0L ); 
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

vb_disp_nonl('Make neighbor member list of original brain (Left): ')
tic
Llist  = vb_find_near_member(xxFL, xxDL, indxL, Rmax);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

%
% --- Make neighbor member list of original brain (Right)
%
tic
vb_disp_nonl('Find neighbor index (Right): ')
[xxDR,xxFR]   = vb_next_distance( F0R, V0R ); 
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

vb_disp_nonl('Make neighbor member list of original brain (Right): ')
tic
Rlist  = vb_find_near_member(xxFR, xxDR, indxR, Rmax);
vb_disp(sprintf('%f[sec]',toc));
%vb_ptime(toc)

%% Whole Brain Parcellation
% Search radious for parcellation (as large as possible)
Rmax = 0.040; % 4cm

% Load ix_area_mod
load(parcel_file);

Nroot_l = length(indxL);
tmp = find(vbmeg_area_ix<=Nroot_l);
half = tmp(end);

% Parcellation
[lh_id] = vb_find_near_member...
  (xxFL, xxDL, indxL(vbmeg_area_ix(1:half)), Rmax);
[rh_id] = vb_find_near_member...
  (xxFR, xxDR, indxR(vbmeg_area_ix(half+1:end)-Nroot_l), Rmax);


%% 
lh_cortex_id = cell(length(lh_id), 1);
rh_cortex_id = cell(length(rh_id), 1);


%% 1st element contains sub cortex indicies
%lh_cortex_id{1} = FS_wm.lh_subcortex_index;
%rh_cortex_id{1} = FS_wm.rh_subcortex_index;

% (lh/rh)_cortex_id contains only cortex index(remove subcortex elements)
for n=1:length(lh_id)
    lh_cortex_id{n} = lh_id{n};
    [tmp, ind] = intersect(lh_id{n}, FS_wm.lh_subcortex_index);
    lh_cortex_id{n}(ind) = [];
end
for n=1:length(rh_id)
    rh_cortex_id{n} = rh_id{n};
    [tmp, ind] = intersect(rh_id{n}, FS_wm.rh_subcortex_index);
    rh_cortex_id{n}(ind) = [];
end

% Save index
save(parcel_file, '-append', 'lh_id', 'rh_id', 'lh_cortex_id', 'rh_cortex_id', 'brain_file');

% Create membership matrix(parcels to brain model(V))
% ix = find(parcel(1, :) ~= 0); means 
[membershipmat] = make_membershipmat(brain_file, parcel_file);
save(parcel_file, '-append', 'membershipmat');

disp(sprintf('\nSaved:%s', parcel_file));

% Check parcellation convers all verticies or not.
is_cover_all = dmri_parcels_cover_all_vertices(parcel_file, ...
                                            brain_file, ...
                                            fs_info_file);
                                        
if is_cover_all == false
    warning(['Parcellation result may not be correct. ', ...
             'There are vertices that are not affiliated ', ...
             'with either parcels. Please try to increase ', ...
             'parcellation Number(current:' num2str(num_vertex) ').']);
end
toc(start);
