function [CC,DD,xyz0,Keys0]=dmri_connections_regions_calc(dmrifile, brainfile, areafile, parcelfile)
%  CALCULATE DMRI CONNECTIONS BETWEEN REGIONS 
% [Usage]
%  [CC,DD,xyz0,Keys0]=dmri_connections_regions_calc(dmrifile, brainfile, areafile);
%
% [Input]
%  dmrifile  : connectivity matrix obtained from DMRI processor
%  brainfile : brainfile of VBMEG
%  areafile  : areafile of coarse region parcellation such as AAL and
%              brodmann. AAL is recommended.
% [Output]
%  CC        : region connectivity matrix (number of connections average over region vertices)
%  DD        : region distance matrix in unit meter  (distance average over region vertices). 
%
% 2015/09/08 O.Yamashita
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if nargin < 4
    parcelfile = [];
end


%---------------------
% brain model
%---------------------

B=load(brainfile);

V = B.V .* 1000; % [m] => [mm]
F0 = B.F.F3;

%------------------------------
%  connectivity
%------------------------------

% IX : parcel indicies to region indices
[tmp,IX,xyz,Keys] = inner_parcels_to_regions(areafile,dmrifile,parcelfile,V);

% remove empty regions
[IX0,xyz0,Keys0] = inner_remove_null_regions(IX,xyz,Keys);  

% connectivity matrix, distance matrix
[C,D] = inner_get_connectivity_matrix(dmrifile,parcelfile);

% region connectivity
[CC,DD] = inner_get_region_connectivity(C,D,IX0);
    

%------------------------------------------------------
% inner functions 
%

function [C,IX,xyz,Keys] = inner_parcels_to_regions(areafile,dmrifile,parcelfile,V)

if isempty(parcelfile)  % new format
    AC = load(dmrifile);
    C = AC.connectivity_matrix;
    ix = AC.vbmeg_area_ix;
else
    AC = load(dmrifile);
    P = load(parcelfile);
    ix = P.ix_area_mod;
    C = full(AC.Ind);

end

% Corse representaion of connectivity matrix (require area file)
A = load(areafile);
Narea = length(A.Area);

for aa = 1 : Narea
    tmpArea = A.Area{aa};
    xyz(aa,:) = mean(V(tmpArea.Iextract,:),1);  % center of region
    [tmp,jx] = intersect(ix, tmpArea.Iextract);
    IX{aa} = jx;
    Keys{aa} = tmpArea.key;
end

function [IX0,xyz0,Keys0] = inner_remove_null_regions(IX,xyz,Keys)
% erase empty area
nn = 1;
for aa = 1 : size(xyz,1)
    if ~isempty(IX{aa})
        xyz0(nn,:) = xyz(aa,:);
        IX0{nn} = IX{aa};
        Keys0{nn} = Keys{aa};
        nn = nn + 1;
    end
end

function [CC,DD] = inner_get_region_connectivity(C,D,IX0)

Narea0 = length(IX0);
CC = zeros(Narea0, Narea0);
DD = zeros(Narea0, Narea0);
for aa1 = 1 : Narea0
    for aa2 = 1 : Narea0
        CC(aa1,aa2) = mean(mean(C(IX0{aa1},IX0{aa2})));
        DD(aa1,aa2) = mean(mean(D(IX0{aa1},IX0{aa2})));
        
    end
end



function [C,D] = inner_get_connectivity_matrix(dmrifile,parcelfile)

if isempty(parcelfile)  % new format
    AC = load(dmrifile);
    C = AC.connectivity_matrix;
    
    D = zeros(size(AC.connectivity_matrix));
    tmp = AC.delay_ms_matrix ;
    ix = find(tmp > 0);
    D(ix) = (tmp(ix) - AC.dmri_parm.tau)/1000*AC.dmri_parm.v;   % distance matrix in s*(m/s) = m   
    
else
    AC = load(dmrifile);
    P = load(parcelfile);
    ix = P.ix_area_mod;
    C = full(AC.Ind);
   
    D = zeros(size(AC.connectivity_matrix));
    tmp = AC.Delta;
    ix = find(tmp > 0);
    D(ix) = (tmp(ix) - AC.dmri_parm.tau)/1000*AC.dmri_parm.v;   % distance matrix in s/(m/s) = m   
    
    
end
