function	[Atlas] = vb_correct_atlas_label_LR(Atlas,brainfile,Para)
% Correct holes and islands in 'brodmann'-atlas label by morphology
%   [Atlas] = vb_correct_atlas_label_LR(Atlas,brainfile)
% --- Input
% Atlas : structure of atlas label
% Atlas.xxP        : label value for each vertex
% Atlas.label      : label value
% Atlas.label_name : label_name corresponding to 'label'
% 
% brainfile
%
% Para.Rlabel : max radius for morphology  [m]
% Para.rate   : number less than rate*(# of largest region) is deleted
% --- Output
% Atlas : structure of atlas label
% --- Method
% extract connected region in left/right cortex sepalately
% fill holes by gaussian filter morphology
%
% 2006-11-12 M. Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~exist('Para','var'), Para = []; end

% max radius for morphology
if isfield(Para,'Rlabel')
	Rmax  = Para.Rlabel;
else
	Rmax  = 0.012;% [m]
end

if isfield(Para,'rate')
	rate  = Para.rate;
else
	rate  = 0.1;
end

% load brain
[V, F] = vb_load_cortex(brainfile);

% Left/Right cortical patch index
NL = F.NdipoleL;
FL = F.F3L;
FR = F.F3R;
NV = size(V,1);	% # of vertex

name  = Atlas.label_name ; % label_name corresponding to 'label'
label = Atlas.label  ; % label value;
Narea = length(label);

% New label to separate L-R
Label = [];
Name  = [];
Lid   = 0;
Rid   = max(label)+10;
Lext  = '_L';
Rext  = '_R';

% label value for each vertex [Nvertex x 1]
xxP   = Atlas.xxP; 
xxQ   = zeros(NV,1);

Num = sum( xxP == 0 );
fprintf('--- Unlabeled vertex = %d\n',Num)
fprintf('--- Find connected region\n')

id = 0;

for n=1:Narea
    % vertex index for n-th label
    ix = find( xxP == label(n) );
    
    if isempty(ix), continue; end;
    
    %
    % --- Left cortex
    %
    % find connected region
    ixL = ix( ix <= NL );
	[Vindx, Nall] = vb_connected_vertex(ixL,FL);

	% set label to largest connected region
	if ~isempty(Vindx)
		jj = find( Nall >= Nall(1)*rate );
		xxQ([Vindx{jj'}]) = label(n) + Lid;
	end
	id = id + 1;
	Label(id) = label(n) + Lid;
	Name{id}  = [name{n} Lext];
	
    %
    % --- Right cortex
    %
    % find connected region
    ixR = ix( ix > NL );
	[Vindx, Nall] = vb_connected_vertex(ixR,FR);

	% set label to largest connected region
	if ~isempty(Vindx)
		jj = find( Nall >= Nall(1)*rate );
		xxQ([Vindx{jj'}]) = label(n) + Rid;
	end
	id = id + 1;
	Label(id) = label(n) + Rid;
	Name{id}  = [name{n} Rext];
	
end

fprintf('--- Morphological smoothing\n')

% New label
Narea = length(Label);

% unlabeled index
ixfix = find( xxQ == 0 );
Nfix  = length(ixfix);

% label weight
PP = zeros(Nfix,Narea);

% smoothing filter
W  = vb_spatial_gauss_filter(brainfile,Rmax,Rmax,[1:NV]);
W  = W(ixfix,:);

for n=1:Narea
    % vertex index for n-th label
    ix = find( xxQ == Label(n) );
    
    if isempty(ix), continue; end;
    
    PP(:,n) = sum(W(:,ix), 2);
	
end

for n=1:Nfix
	[Pmax, id] = max(PP(n,:));
	
	if Pmax > 0,
		xxQ(ixfix(n)) = Label(id);
	end
end

Atlas.xxP   = xxQ; % label value for each vertex
Atlas.label =  Label; % label value
Atlas.label_name =  Name ; % label_name corresponding to 'label'

Num = sum( xxQ == 0 );
fprintf('--- Final unlabeled vertex = %d\n',Num)

%%% surface morphology
%	Iextract = vb_open_area(Jarea, R, nextIX, nextDD);
%	Iextract = vb_close_area(Jarea, R, nextIX, nextDD);
