function basis = vb_calc_lead_field(V, xx, pick, Qpick, Wsensor, bem_mode, ...
                                BEM, Vhead, Fhead, Omega, Sout, SPHinfo);
% Leadfield calculation
%
% --- INPUT
%  V(n,:)  : dipole position (3-D coordinate) at n-th vertex
% xx(n,:)  : dipole current direction (unit vector) at n-th vertex
%  pick(k, 1:3) : Sensor coil position  : , 
% Qpick(k, 1:3)	: Sensor coil direction : 
%
% Wsensor(m,n) = n-th coil weight for m-th sensor channel
%    basis(channel,dipole) = Wsensor * basis(coil,dipole)
%
% bem_mode: Leadfield calculation mode
%  MEG
%      0  : MEG (Sphere model:Sarvas) ǥ
%      1  : MEG (BEM)                 ˡ
%      2  : MEG (Spherical Harmonics) ̴ؿŸ
%  EEG
%      3  : EEG (Sphere model)        3صǥ
%      4  : EEG (BEM)                 ˡ
%
%   BEM      : BEM parameters
%   Vhead    : Vertex on boudary surface    ̤κɸ
%   Fhead    : triangle of boudary surface  ĺǥå
%   Omega    : Solid angle matrix           ̷
%   Sout     : Normal of boudary surface    ̤γˡ
%	SPHinfo  : Parameters for Spherical Harmonics
% 
% --- OUTPUT
% basis( L * Nvertex , Npick) 
%   : Magnetic field for MEG case
%   : Electric potential for EEG case
% basis( n, k )
%  : k-th sensor magnetic field for dipole current at n
%  : k-th sensor electric potential for dipole current at n
%
% written by M. Sato  2005-8-8
% modified by T. Sako 2006-8-23
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% Basis mode definition
global vbmeg_inst;
const = vbmeg_inst.const;
% const.BASIS_MEG_SPHERE;   %  MEG Sphere model (Sarvas)
% const.BASIS_MEG_BEM;      %  MEG BEM
% const.BASIS_MEG_HARMONICS;%  MEG Spherical harmonics expansion
% const.BASIS_EEG_SPHERE;   %  EEG 3-shell Sphere model
% const.BASIS_EEG_BEM;      %  EEG 3-shell BEM

NV      = size(V,1);
Npick   = size(pick,1);
Nsensor = size(Wsensor,1);

%
% --- Leadfield calculation (꡼ɥեɹ׻)
%
tic;

switch bem_mode
case const.BASIS_MEG_SPHERE, 
	%  MEG Sphere model (Sarvas)
	fprintf('--- MEG Sphere model (Sarvas) ');

	BB = zeros(NV , Npick);
	
	for i=1:Npick,
		BB(:,i) = vb_sarvas_new(V, xx, pick(i,:), Qpick(i,:) );
	end
	
	%basis = BB(:,1:Nsensor) - BB(:,(Nsensor+1):end);
	basis = BB * Wsensor';

case const.BASIS_MEG_BEM, 
	%  MEG BEM
	fprintf('--- MEG BEM-linear-Galerkin ');

	BB = vb_bem_meg(Omega,Sout,Vhead,Fhead,V,xx,pick,Qpick);
	
	%basis = BB(1:Nsensor,:) - BB((Nsensor+1):end,:);
	basis = Wsensor * BB;
	basis = basis';

case const.BASIS_MEG_HARMONICS, 
	%  MEG Spherical harmonics expansion
	fprintf('--- MEG Spherical Harmonics expansion ');
	N    = SPHinfo.func_order;
  	Rmax = SPHinfo.radius;
  	A    = SPHinfo.harmo_coef;

  	BB = vb_spherical_harmo_magnetic_field(V, xx, N, Rmax, pick, Qpick, A);
	
	%basis = BB(:,1:Nsensor) - BB(:,(Nsensor+1):end);
  	basis = BB * Wsensor';

case const.BASIS_EEG_SPHERE, 
	% EEG 3-shell Sphere model
	fprintf('--- EEG potential (Sphere model) ');
	
	Rmax  = max(sqrt(sum(V.^2,2)));
	Rpick = mean(sqrt(sum(pick.^2,2)));
	R     = BEM.R * Rpick;
	
	% Electric potential
	basis = vb_eeg_legendre(V, xx, pick, R, BEM.sigma);

case const.BASIS_EEG_BEM, 
	%  EEG 3-shell BEM
	fprintf('--- EEG potential (BEM) ');
	
	% Electric potential
	basis = vb_bem_eeg(Omega,Sout,Vhead,Fhead,V,xx,pick,BEM);
	
	basis = basis';
end

fprintf('%f[sec]\n',toc);
