function eeg = vb_eeg_multi_shell(P, Q, Robs, R, sigma, NJ)
% EEG potential for dipoles in multilayer spherical model 
%        eeg = vb_eeg_multi_shell(P, Q, Robs, R, sigma)
% INPUTS 
% P    : current dipole position (in meters)          ( NP x 3 )
% Q    : current dipole moment (direction)            ( NP x 3 )
% Robs : EEG sensors (in meters) on the scalp         ( NS x 3 )
% R    : Radii of sphere from innermost to outermost  ( NL x 1 )
% sigma: Conductivity    from innermost to outermost  ( NL x 1 )
% NJ   : # of Berg Dipoles     (Default: 3)
%
% OUTPUT
%   eeg : EEG Lead field matrix   ( NP x NS )
%
% Single Dipole in a multilayer sphere can be approximated 
%        by multiple dipoles in a single shell. 
% Z. Zhang,  Phys. Med. Biol., vol.40, pp. 335-349, 1995.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

NL	 = length(R);		% # of sphere layers
NV	 = size(P,1);		% # of dipoles
Nobs = size(Robs,1);	% # of EEG sensors

sigmaL  = sigma(NL);		% Conductivity in outermost sphere
eeg     = zeros(NV,Nobs);	% EEG Lead field matrix

if NL==1,
	for n=1:Nobs
		Rn   = Robs(n,:);		% n-th EEG sensor position
		eeg(:,n) = vb_eeg_one_shell(P, Q, Rn, sigmaL);
	end
	
	return
end

if nargin < 6  
    % Default # of Berg dipoles
    NJ=3;       
end

%  mu: Berg radial factors            ( NJ x 1 )
% lam: Berg magnitude factors         ( NJ x 1 )
[mu,lam] = vb_berg_param(R, sigma, NJ);

for n=1:Nobs
	Rn   = Robs(n,:);		% n-th EEG sensor position
	eeg0 = zeros(NV,1);
	
	% Approximation by NJ dipoles in a single shell
	for j=1:NJ
		eeg0 = eeg0 + vb_eeg_one_shell(mu(j)*P, lam(j)*Q, Rn, sigmaL);
	end;
	
	eeg(:,n) = eeg0;
end

