function [mu, lam, f, fval, flag, nsteps] = ...
         vb_berg_param(R, sigma, NJ, Nmax, Nopt, mu_ini, lam_ini)
% Berg Parameter calculation for EEG multilayer spherical forward model 
%   [mu,lam]   = vb_berg_param(R, sigma)
%   [mu,lam,f] = vb_berg_param(R, sigma, NJ, Nmax, Nopt)
%
% Computes the Berg parameters:
% 
% INPUTS (Required):
%    R    : Radii of sphere from innermost to outermost    ( NL x 1 )
%    sigma: Conductivity    from innermost to outermost    ( NL x 1 )
%
% INPUTS (Optional):
%   Nmax : # of Legendre Expansion 
%            to fit Berg Parameters   (Default: 200)
%     NJ : # of Berg Dipoles          (Default: 3)
%   Nopt : # of optimization search
%  mu_ini: Initial value for radial factors           ( NJ x 1 )
% lam_ini: Initial value for magnitude factors        ( NJ x 1 )
%  
%  NL = # of sphere layers; 
%  NJ = # of Berg Dipoles
%
% OUTPUTS:
%    mu: Berg radial factors            ( NJ x 1 )
%   lam: Berg magnitude factors         ( NJ x 1 )
%     f: Legendre Expansion Weights used to fit Berg Parameters ( Nmax x 1 )
%
%
% Single Dipole in a multilayer sphere can be approximated 
%        by multiple dipoles in a single shell. 
% Z. Zhang,  Phys. Med. Biol., vol.40, pp. 335-349, 1995.
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% # of sphere
NL = length(R);

if nargin < 3  
    % Default # of Berg Parameters 
    NJ=3;       
end

if nargin < 4    
    % Default # of Legendre expansion
    Nmax = 200;  
end

if nargin < 5    
    % Default # of iteration in optimum search
    Nopt = 1000;  
end

if nargin < 6
    % Default Value for Initial Eccentricity Parameters 
    mu_ini = (1/(NJ+2))*[1:NJ];  
    % Default Value for Initial Magnitude Parameters
    lam_ini = 0.2*ones(1,NJ);      
end

%
% Set minimization option
%
%OPTIONS = optimset('MaxFunEvals',  NJ*Nopt, ...
%                   'MaxIter',      NJ*Nopt, ...
%				   'TolFun', [0  1.e-9  1.e-9]); 

OPTIONS = optimset('MaxFunEvals',  NJ*Nopt, ...
                   'MaxIter',      NJ*Nopt, ...
				   'TolFun', [0  1.e-8  1.e-8]); 

% Computes the weights f(n) for legendre expansion

f = vb_multi_shell_param(R, sigma, Nmax);

%
% Minimize the error function defined by Zhang
%
    
[berg, fval, flag, outputs]= ...
    fminsearch('vb_berg_error',[mu_ini lam_ini(2:NJ)], OPTIONS, R, f); 

nsteps = outputs.iterations;

mu        = berg(1:NJ);
lam(2:NJ) = berg(NJ+1:2*NJ-1);
lam(1)    = f(1) - sum(lam(2:NJ));
