function vb_job_leadfield_extra_eeg(varargin)
% EEG Lead field matrix for extra dipole using BEM
% 
% --- Syntax
% vb_job_leadfield_extra_eeg(extra_basis_parm)
% vb_job_leadfield_extra_eeg(proj_root, extra_basis_parm)    [old style]
%
% --- Input
% Fields of extra_basis_parm
% .basis_file: Leadfield matrix is saved as this file
% .eeg_file  : to load sensor position 'pick'
% .mps_file  : Extra dipole model file (.mps.mat)
% .head_file : Head shell model file for BEM
% .sigma     : Conductivity  from innermost to outermost
% .Recalc    : (Optional) re-calculation flag for solid angle matrix 'Omega'
%   if 'Omega' file exist, time consuming 'Omega' calculation is not done
%   if Recalc==ON, 'Omega' is re-calculated even of 'Omega' file exist
%
% --- Output
% Basis file
% basis( Norient * Nvertex , Nsensor)
% 	  basis( n, k )   : k-th sensor field for dipole current at n
%
% --- History
% 2011-09-06 Masa-aki Sato

if length(varargin) == 1
  proj_root = [];
  extra_basis_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  extra_basis_parm = varargin{2};
end

% Filename
mpsfile = fullfile(proj_root, extra_basis_parm.mps_file);
eegfile = fullfile(proj_root, extra_basis_parm.eeg_file);

% Prepare extra dipole information
load(mpsfile,'Pointlist');
extra_pos = [];
extra_direction = [];

%%%% ---- This part may be changed ---- %%%%
for i=1:length(Pointlist)
  % Three dipoles at each position
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_direction = [extra_direction; 1 0 0];
  extra_direction = [extra_direction; 0 1 0];
  extra_direction = [extra_direction; 0 0 1];
end

% Load EEG sensor 
[pick] = vb_load_sensor(eegfile);

% sigma  : Conductivity  from innermost to outermost
sigma = extra_basis_parm.sigma;

if iscell(extra_basis_parm.head_file)
	headfile = extra_basis_parm.head_file;
	joinfile = [vb_join_cell_file_name(headfile) '.head.mat'];
	joinpath = fullfile(proj_root, joinfile);
	
	if ~exist(joinpath,'file')
		sigma=[];layer=[];mode = 3; % MAX mode for radius
		vb_head_join_files(headfile,proj_root,joinpath, sigma,layer,mode);
	end
	extra_basis_parm.head_file = joinfile;
end

new_headfile = fullfile(proj_root, extra_basis_parm.head_file);

omega_file   = [vb_get_basename(new_headfile,'.head.mat') '.omega.mat'];

fprintf('Load %s\n',new_headfile)
load(new_headfile);
% Vhead : (# of vertex) x 3
% Fhead : (# of patch) x 3
% Nvertex : (# of surface) x 2
% Npatch  : (# of surface) x 2
% Sigma   : 1 x (# of surface)

fprintf('--- BEM for EEG\n')
fprintf('# of vertices = %d\n',size(Vhead,1));

if ~exist('Nvertex', 'var'), Nvertex = [1 length(Vhead)]; end;
if ~exist('Npatch', 'var'),  Npatch  = [1 length(Fhead)]; end;

Nsurf = size(Nvertex,1);
fprintf('# of surfaces = %d\n',Nsurf);

BEM.sigma   = [sigma(1:Nsurf) 0];
BEM.Nvertex = Nvertex;
BEM.Npatch  = Npatch;

if exist(omega_file,'file')
	fprintf('Load %s\n',omega_file)
	load(omega_file)
end

if  ~exist('Omega','var') || extra_basis_parm.Recalc == ON,
	tic;
	fprintf('--- Solid angle calculation \n');
		
	% (Normal vectors (XXhead) are outwarded here)
	[Omega, Sout] = vb_solid_angle_grk(Vhead,Fhead,XXhead);

	vb_ptime(toc);
	vb_fsave(omega_file,'Omega', 'Sout');
	fprintf('Saved omega file: %s\n',omega_file);
end

tic;

% Multilayer model correction
fprintf('--- Multisurface correction\n');
[Omega] = vb_bem_matrix(Omega,BEM);

% Self-weighting factor
%   : Add integrating result of the basis function on the triangular surface.
fprintf('--- Auto solid angle calculation\n');
Omega = vb_solid_auto_grk(Omega,Fhead,Sout);

vb_ptime(toc);

fprintf('--- EEG potential (BEM) \n');
	
% Electric potential
basis = vb_bem_eeg( Omega, Sout, Vhead, Fhead, ...
		extra_pos, extra_direction, pick, BEM );
basis = basis';

% Save data
vb_fsave(fullfile(proj_root, extra_basis_parm.basis_file),'extra_basis_parm','basis', ...
         'Pointlist');
