function vb_job_leadfield_extra(varargin)
% Lead field matrix for extra dipole (Biot-Savart)
% 
% [syntax]
% vb_job_leadfield_extra(extra_basis_parm)
% vb_job_leadfield_extra(proj_root,extra_basis_parm)    [old style]
%
% [input]
% proj_root       : <<string>> VBMEG project root directory. 
% extra_basis_parm: <<struct>> Parameters for leadfield calculation.
% --- fields of extra_basis_parm
%  .meg_file  : <<string>> Load 'pick' and 'Vcenter' from this file.
%  .mps_file  : <<string>> Extra dipole model file (.mps.mat)
%  .basis_file: <<string>> Leadfield matrix is saved as this filename. 
%  .sensortype: <optional> <<string>> This value is compared with
%               MEGinfo.ChannelInfo.Type.
% ---
%
% [output]
% Leadfield file for extra-dipole (.basis.mat) is created. It have the
% same file with that for cortical surface model. 
%
% [example]
% >> proj_root = 'your_project_root_directory';
% >> extra_basis_parm.meg_file   = './testdata/sbj_test_UR.meg.mat';
% >> extra_basis_parm.mps_file   = './testdata/sbj_test_UR_eyes.mps.mat';
% >> extra_basis_parm.basis_file = './testdata/sbj_test_UR_eyes.basis.mat';
% >> vb_job_leadfield_extra(proj_root,extra_basis_parm);
%
% [history]
% 2008-06-27 Taku Yoshioka
% 2008-10-15 Taku Yoshioka
%  Support '.sensortype' parameter
% 2011-02-28 taku-y
%  [debug] 'basis_file' is interpreted as a relative path from
%  'proj_root'. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if length(varargin) == 1
  proj_root = [];
  extra_basis_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  extra_basis_parm = varargin{2};
end


proj_root = vb_rm_trailing_slash(proj_root);

% Filename
mpsfile = fullfile(proj_root, extra_basis_parm.mps_file);
megfile = fullfile(proj_root, extra_basis_parm.meg_file);

% Prepare extra dipole information
load(mpsfile,'Pointlist');
extra_pos = [];
extra_direction = [];
for i=1:length(Pointlist)
  % Three dipoles at each position
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_pos = [extra_pos; Pointlist{i}.point];
  extra_direction = [extra_direction; 1 0 0];
  extra_direction = [extra_direction; 0 1 0];
  extra_direction = [extra_direction; 0 0 1];
end

% Load MEG sensor
[pick, Qpick, Wsensor, V0] = vb_load_sensor(megfile);
% Wsensor(m,n) = n-th coil weight for m-th channel

% Select channel (2008-10-15 Taku Yoshioka)
if isfield(extra_basis_parm,'sensortype')
  sensortype = extra_basis_parm.sensortype;
  if ~isempty(sensortype), 
    MEGinfo = vb_load_measurement_info(megfile);
    ix_sensor = find(MEGinfo.ChannelInfo.Type==sensortype);
    ix_sensor2 = [ix_sensor; ix_sensor+length(ix_sensor)];
    pick = pick(ix_sensor2,:);
    Qpick = Qpick(ix_sensor2,:);
    Wsensor = Wsensor(ix_sensor,ix_sensor2);
  end
end

% Make extra dipole leadfield 
B = vb_dipole_magnetic(extra_pos, extra_direction, pick, Qpick );
basis = B * Wsensor';

% Save data
vb_fsave(fullfile(proj_root, extra_basis_parm.basis_file), ...
         'extra_basis_parm','basis','Pointlist');
