function [V,F,xx,BV_index,Vinfo] = ...
    vb_job_brain(varargin)
% Convert FreeSurfer data to VBMEG format.
% - VBMEG Cortical model file(.brain.mat)
% - VBMEG Cortical area file(.area.mat)
% - VBMEG Cortical act file(.act.mat)
%
% [syntax]
% vb_job_brain(brain_parm)
% vb_job_brain(proj_root,brain_parm)  [old style]
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% brain_parm: <<struct>> Parameters for creating cortical surface model
% --- fields of brain_parm
%  analyze_file: <<string>> MRI filename (.nii/.hdr)
% brain_parm.FS_left_file         = [fs_dir '/bem/lh.smoothwm.asc'];
% brain_parm.FS_right_file        = [fs_dir '/bem/rh.smoothwm.asc'];
% brain_parm.FS_left_infl_file    = [fs_dir '/bem/lh.inflated.asc'];
% brain_parm.FS_right_infl_file   = [fs_dir '/bem/rh.inflated.asc'];
% brain_parm.FS_left_curv_file    = [fs_dir '/bem/lh.curv.asc'];
% brain_parm.FS_right_curv_file   = [fs_dir '/bem/rh.curv.asc'];
% 
% brain_parm.FS_left_label_file   = [fs_dir '/label/lh.cortex.label'];
% brain_parm.FS_right_label_file  = [fs_dir '/label/rh.cortex.label'];
%
% brain_parm.FS_left_sphere_file  = [fs_dir,  'lh.sphere.reg.asc'];
% brain_parm.FS_right_sphere_file = [fs_dir,  'rh.sphere.reg.asc'];
% brain_parm.FS_sphere_key        = 'sphere.reg';
% brain_parm.registration_mode='FS'; % FreeSurfer sphere registration.
%  
%  brain_file  : <<string>> Cortical surface model file (.brain.mat).
%  act_file    : <<string>> Cortical activity map file (.act.mat).
%  area_file   : <<string>> Cortical area file (.area.mat).
%  Nvertex     : <<int>>    Total number of vertices for cortical surface model
%                           Priority: Nvertex > reduce_ratio
%  Rmax        : <<float>>  Maximum radius for neighbor search 
%  N_step      : <<int>>    Division number of z axis in matching between
%                           cortical surface model and MRI points
%  display     : <<int>>    Display progress for this steps
%  brain_space : <<string>> cortical model map mode.
%                           ='mni'  : normalize to MNI space 
%                           ='subj' : individual space
% ---
% 
% [output]
% The following files are created: 
%  VBMEG cortical surface model file (.brain.mat)
%  VBMEG area file (.area.mat)
%  VBMEG activity map file (.act.mat)
%
% The following variables are stored in .brain.mat file: 
%
% Standard brain information.
% V         : Cortical vertex point cordinate (SPM_Right_m) 
% F         : Patch index structure
% xx        : Normal vector to cortical surface 
% xxA       : area asigned for each vertex ([m^2])
%
% BV_index  : original vertex index in BV/FS corresponding to brain
% Vinfo     : Vertex dimension structure
%
% xxF{i}    : Next-neighbor index for the vertex-i
% xxD{i}    : Distance between next-neighbor and the vertex-i
% xxT{n}    : vertex index for n-th triangle
% nextIX{i} : Neighbor index list for the vertex-i
% nextDD{i} : Distance from the vertex-i
% normal_stat
%       .neighbor_org             : BV original brain neighbor for specified vertex
%       .normal_stat.normal_org   : BV original brain normal vector
%
% subj      : Individual brain information(structure)
%    .V     : V(n, :) and subj.V(n, :) has same MNI coordinate value.
%    .F     : currently empty(not supported for plotting).
%    .xx
%    .xxA
%    .BV_index
%    .Vinfo
%    .nextIX
%    .nextDD
%    .normal_stat
% 
% The following areas are automatically created and stored in .area.mat
% file: 
% areakey = 'Cortex'         : cortex region without corpus area
%
% [history]
% Ver.2.0 New version for group analysis
% 2015-12-27 M. Sato 
% 2017-03-10 M. Sato added some subject elements(subj.xxA, subj.nextIX, subj.nextDD)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if length(varargin) == 1
  proj_root = [];
  brain_parm = varargin{1};
elseif length(varargin) == 2
  proj_root = varargin{1};
  brain_parm = varargin{2};
end

if ~isfield(brain_parm, 'brain_space')
    brain_parm.brain_space = 'mni';
end

switch(lower(brain_parm.brain_space))
    case 'mni'
        vb_disp('Cortex map mode : MNI');
    case 'subj'
        % create ver1 format BRAIN-MAT file.
        vb_disp('Cortex map mode : Individual');
        vb_job_brain_subj(proj_root, brain_parm);
        return;
    otherwise
        error('Unknown brain_space ''%'' was specified.', brain_parm.brain_space);
end

global  vbmeg_inst
const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

if ~exist('DEBUG_MODE','var'), DEBUG_MODE = 0; end

% Define constant
define = vbmeg_inst.const;

proj_root = vb_rm_trailing_slash(proj_root);

%
% Do not modify following lines
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

% Output file path
if isempty(proj_root)
    brain_file = brain_parm.brain_file;
    area_file  = brain_parm.area_file;
    act_file   = brain_parm.act_file;
else
    brain_file = fullfile(proj_root, brain_parm.brain_file);
    area_file  = fullfile(proj_root, brain_parm.area_file);
    act_file   = fullfile(proj_root, brain_parm.act_file);
end

% ID for MRI image data
analyze_file = brain_parm.analyze_file;
[udir, fname] = vb_get_file_parts(analyze_file);
MRI_ID = vb_mMD5(analyze_file);

%-------------------------------------------------------
% Check registration_mode
%-------------------------------------------------------
flag_spm = (isfield(brain_parm, 'spm_normalization_file') ...
       &&    ~isempty(brain_parm.spm_normalization_file)) ;
       
flag_fs = (isfield(brain_parm, 'FS_left_sphere_file') ... 
       &&    ~isempty(brain_parm.FS_left_sphere_file) ...
       &&  isfield(brain_parm, 'FS_right_sphere_file') ...
       &&   ~isempty(brain_parm.FS_right_sphere_file) ...
       &&  isfield(brain_parm, 'FS_sphere_key') ...
       &&   ~isempty(brain_parm.FS_sphere_key));

if ~isfield(brain_parm, 'registration_mode')
    brain_parm.registration_mode = [];
    
    if flag_fs == 0 && flag_spm == 1,
        brain_parm.registration_mode='SPM';
    end
    if flag_fs == 1 && flag_spm == 0,
        brain_parm.registration_mode='FS';
    end
end

% Reference standard brain
[std_parm, std_brain_dir] = vb_set_icbm152(brain_parm.Nvertex);
std_brain = fullfile(std_brain_dir, std_parm.brain_file);

% List up standard brain resolution
parent_dir = vb_get_file_parts(std_brain_dir);
d = dir(fullfile(parent_dir, '*_*'));
fprintf('Available standard brain model resolution : ')
for k=1:length(d)
    if(d(k).isdir == false)
        continue;
    end
    ix = findstr(d(k).name, '_');
    resolution = d(k).name(ix(end)+1:end);
    fprintf('%s ', resolution);
    if k == length(d), fprintf('\n'); end
end
if exist(std_brain, 'file') ~= 2
    % error case
    fprintf('Specified resolution(Nvertex=%d) cannot be created.\n', brain_parm.Nvertex);
    fprintf('If you use custom resolution, please make new standard brain file as below.\n');
    fprintf('==========\n');
    fprintf('[parm, proj_root] = vb_set_icbm152(%d);\n', brain_parm.Nvertex);
    fprintf('make_std_brain_model(proj_root, parm);\n');
    fprintf('==========\n');
    fprintf('Then re-run this function.\n');
    error('Stop processing.');
end

%-------------------------------------------------------
% Make cortex data from MNI standard brain for group analysis
% 1. Map MNI standard brain to subject brain
% 2. Find nearest point from subject cortical surface 
%    extracted by FreeSurfer/BV
%-------------------------------------------------------
registration_mode = upper(brain_parm.registration_mode);
switch  registration_mode
case    'SPM'
    if flag_spm == 0 || flag_fs == 1,
        error('Please specify SPM normalization file');
    end;
    
    [V,F,xx,BV_index,Vinfo,Vext,Fext] = vb_make_brain_mni(brain_parm, 'SPM');
case    'FS'
    if flag_fs == 0 || flag_spm == 1,
        error('Please specify FS spherical file');
    end;
    
    [V,F,xx,BV_index,Vinfo,Vext,Fext] = vb_make_brain_mni(brain_parm, 'FS');
otherwise
    error('Registration mode error');
end

if DEBUG_MODE==2, return; end;
%----------------------------------------------
% Save cortical surface model
%----------------------------------------------

vb_disp(['Save brain model '], VERBOSE_LEVEL_NOTICE);
vb_disp(sprintf('     filename = %s', brain_file), ...
        VERBOSE_LEVEL_NOTICE);
[p, n, e] = vb_get_file_parts(brain_file);
if ~isempty(p) && ~exist(p, 'dir')
    vb_mkdir(p);
    vb_disp(['Create directory: ' p]);
end
copyfile(std_brain, brain_file);


%----------------------------------------
% Search next-point index and distance
%----------------------------------------
vb_disp('Search next-point index and distance ', ...
        VERBOSE_LEVEL_NOTICE);
        
% neighbor index for model vertex (V)
%[xxD, xxF, xxT, xxN] = vb_next_distance( F.F3, V ); % unit : m
%Nzero = length(find( xxN == 0));
%fprintf('# of No neibor point = %d (# of vertex = %d)', Nzero, length(xxN))

% neighbor index for extended vertex (Vext)
[xxD, xxF, xxT, xxN] = vb_next_distance( Fext.F3, Vext ); % unit : m

%----------------------------------------------
% Calculate area assigned to the Vext
%----------------------------------------------
%tic; 
[xxA_all] = vb_calc_patch_area(Vext, Fext.F3, xxT);
%vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

if DEBUG_MODE==1, return; end
%----------------------------------------------
% Search neighbor points along cortex sheet
%----------------------------------------------
Rmax    = brain_parm.R_max; 
Display = brain_parm.display;

% total number of vertex
Ndipole = Vinfo.Ndipole;
Vindx   = 1:Ndipole;
Vindx_ext   = 1:size(Vext,1);

[nextIXext , nextDDext] = ...
	vb_find_neighbor_all(Rmax, xxF, xxD, Vindx_ext, Display);

% neighbor index for V
[nextIX, nextDD] = vb_reduce_neighbor_data(Vindx,nextIXext,nextDDext);

%  asign external vertex by base vertex according to distance
[ext_index,ext_table] = ...
	vb_asign_neighbor_vertex(Vindx,nextIXext,nextDDext);
	
%  asign external vertex area to base vertex area according to distance
xxA = vb_asign_ext_area(xxA_all, Vindx, ext_index, ext_table);



%--------------------------------------------------------------------------
% Save FS spherical coordinate
%--------------------------------------------------------------------------
% if flag_fs==1,
%     vb_disp(['Save FS spherical coordinate '], VERBOSE_LEVEL_NOTICE);
%     
%     parm.brain_file = brain_parm.brain_file;
%     parm.brain_sphR = brain_parm.FS_right_sphere_file;
%     parm.brain_sphL = brain_parm.FS_left_sphere_file;
%     parm.key        = brain_parm.FS_sphere_key;
%     vb_job_brain_add_sphcoord(proj_root, parm);
% end

%--------------------------------------------------------------------------
% Copy two area files derived from two anatomical divisons
%--------------------------------------------------------------------------
    
vb_disp(['Copy area files for Atras'], VERBOSE_LEVEL_NOTICE);
vb_copy_std_atlas_area(brain_file, brain_parm.Nvertex);

%--------------------------------------
% Make new area and act files
%--------------------------------------
act_new.key     = 'Uniform';
act_new.xxP     = ones(Ndipole,1);
act_new.comment = 'homogeneous fMRI constraint';

vb_add_act([act_file], act_new, MRI_ID, OFF);

% Make Area file
Vindx  = [Vinfo.cortexL; Vinfo.cortexR];
AreaNew.key      = 'Cortex';
AreaNew.Iextract = Vindx; %[1:Ndipole]'; 

vb_add_area(area_file, AreaNew, MRI_ID, OFF);

if DEBUG_MODE==1, return; end;

%---------------------------------------------------
% Find subject original vertex near the reduced cortical vertex
%---------------------------------------------------
normal_stat = vb_original_normal_statics(proj_root,brain_parm,BV_index);

% normal average over neighbor points of original brain
xx = vb_calc_normal_average(normal_stat.neighbor_org, normal_stat.normal_org);

vb_disp('Save neighbor statics in the original brain', ...
        VERBOSE_LEVEL_NOTICE);
    
subj = struct;
subj.BV_index = BV_index;
subj.MRI_ID   = MRI_ID;
subj.Vinfo = Vinfo;
subj.V        = V;
subj.F        = [];
subj.xx       = xx;
subj.xxA      = xxA;
subj.xxD      = xxD;
subj.xxF      = xxF;
subj.nextIX   = nextIX;
subj.nextDD   = nextDD;
subj.normal_stat = normal_stat;
vb_save(brain_file, 'subj', 'brain_parm');

%-----------------------------------------
% Check brain model by 3D & MRI image
%-----------------------------------------
sub_fig  = vb_check_brain_model(proj_root,brain_parm,Vext,Fext);
fig_file = strrep(fullfile(proj_root, brain_parm.brain_file),...
                  '.brain.mat', '.png');
vb_savefig_as_shown(sub_fig, fig_file);

%--------------------------
% Save project file
%--------------------------
proj_file = get_project_filename;
if ~isempty(proj_file)
    project_file_mgr('load', proj_file);
    project_file_mgr('add', 'brain_parm', brain_parm);
end

