function	[V,F,xx,brain_parm] = vb_job_smooth_cortex(proj_root,brain_parm)
% Make smooth cortex from brain model by morphological smoothing
%  [V,F,xx,brain_parm] = vb_job_smooth_cortex(proj_root,brain_parm)
% --- Input file name
% proj_root
% brain_parm.brain_file  =  [ base_brain_file_name '.brain.mat']
% brain_parm.area_file   =  [ base_area_file_name  '.area.mat' ]
% brain_parm.act_file    =  [ base_act_file_name   '.act.mat'  ]
% brain_parm.analyze_file
% --- Optional Input
% brain_parm.Nvertex  = 3000 (Default) : Number of vertex 
% brain_parm.R_fill   = 4 : radius to fill inside of sulcus
% brain_parm.R_smooth = [ 6 6 -6 -6] : radius for morphological smoothing
% brain_parm.vstep    = 2 : Subsampling step for mask image [mm]
% Para.Nloop    = 200: iteration number for expanding sphere to brain surface
% Para.Nlast    = 0:   iteration number for additional smoothing
% 
% ---- Output file name
% brain_parm.brain_file = [ base_brain_file_name '_smooth.brain.mat']
% brain_parm.area_file  = [ base_area_file_name  '_smooth.area.mat' ]
% brain_parm.act_file   = [ base_act_file_name   '_smooth.act.mat'  ]
% ---- Output (also saved in the brain_file)
% V      : Cortical vertex point cordinate (SPM_Right_m) 
% F      : Patch index structure
% xx     : Normal vector to cortical surface 
% ---- Saved variable in the brain_file
% xxA       : area for vertex
%
% xxF{i}    : Next-neighbor index for the vertex-i
% xxD{i}    : Distance between next-neighbor and the vertex-i
% xxT{n}    : vertex index for n-th triangle
% nextIX{i} : Neighbor index list for the vertex-i
% nextDD{i} : Distance from the vertex-i
% Original_info.parent_ix  : Original vertex index corresponding to new vertex
% Original_info.brain_file : Original brainfile;
%
% Vinfo     : Vertex dimension structure
%
% M. Sato 2006-7-21
% M. Sato 2007-3-16
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

DEBUG_MODE=0;

proj_root = vb_rm_trailing_slash(proj_root);

brain_file = brain_parm.brain_file;
area_file  = brain_parm.area_file ;
act_file   = brain_parm.act_file  ;

add_name   = '_smooth';

brain_parm.brain_file  = vb_change_file_basename(brain_file,add_name);
brain_parm.area_file   = vb_change_file_basename(area_file,add_name);
brain_parm.act_file    = vb_change_file_basename(act_file,add_name);

% Original input brain_file
original_brainfile = [proj_root filesep brain_file];

% Output file path
brain_file = [proj_root filesep brain_parm.brain_file];
area_file  = [proj_root filesep brain_parm.area_file ];
act_file   = [proj_root filesep brain_parm.act_file  ];

[V,F] = vb_load_cortex(original_brainfile);  

% ID for MRI image data
[udir, fname] = fileparts(brain_parm.analyze_file);
MRI_ID = fname;

% Subsampling step for mask image [mm]
if ~isfield(brain_parm,'vstep'), brain_parm.vstep = 2; end;

%
%----- ɸѴ: Spm_right [m] -> Analyze_right [mm]
%
disp(['--- Change coordinate to Analyze_right [mm] ']);

[Vdim, Vsize] = analyze_hdr_read(brain_parm.analyze_file);
Vox = vb_spm_right_to_analyze_right_mm(V, Vdim, Vsize);

% Calculate mask image size
brain_parm.Dim   = vb_mask_image_size(Vdim,Vsize, brain_parm.vstep);

%
% --- Make smoothed cortex coordinate from original cortex
%

[Vox,F,xx,original_indx,Vinfo] = vb_smooth_cortex(Vox, F, brain_parm);

% 
% --- Back to original coordinate
%  Vox = V*1000 - Vorgin;
%  V   = (Vox + Vorgin) * 0.001;
disp(['--- Back to Spm_right [m] coordinate ']);

V = vb_analyze_right_mm_to_spm_right(Vox,Vdim,Vsize);
Vinfo.Coord = 'SPM_Right_m';

if DEBUG_MODE==2, return; end;

%
% --- Search next-point index and distance
%
tic;
fprintf('--- Search next-point index and distance ');

[xxD, xxF, xxT] = vb_next_distance( [F.F3L ; F.F3R] , V ); % unit : m

fprintf('%f[sec]\n',toc);

%
% --- Calculate area assigned to the vertex
%
tic

[xxA] = vb_calc_patch_area(V, F.F3, xxT);

fprintf('%f[sec]\n',toc);

original_info.parent_ix  = original_indx;
original_info.brain_file = original_brainfile;

fprintf(['--- Save brain model \n']);
fprintf('     filename = %s\n', brain_file );

vb_save([brain_file], 'F','V','xx','xxF','xxD','xxA',...
     'Vinfo','MRI_ID','original_info');

% total number of vertex
Ndipole = Vinfo.Ndipole;
Vindx   = 1:Ndipole;

% Make activity map file
act_new.key     = 'Uniform';
act_new.xxP     = ones(Ndipole,1);
act_new.comment = 'artificial data';

vb_add_act([act_file], act_new, MRI_ID, OFF);

% Make Area file
AreaNew.key      = 'Cortex';
AreaNew.Iextract = [1:Ndipole]'; 

vb_add_area([area_file], AreaNew, MRI_ID, OFF);

% Check brain model by 3D & MRI image
vb_check_brain_model(proj_root,brain_parm);

if DEBUG_MODE==1, 
	figure;
	hist(xxA*1e6, 0:0.25:10);
	xlim([0 10])
	title('Histgram of patch area')
	xlabel('Area')
	return; 
end;
%
% Search neighbor points along cortex sheet
%

if isfield(brain_parm,'R_max'),
	Rmax    = brain_parm.R_max; 
else
	Rmax    = 18e-3; 
end
if isfield(brain_parm,'R_max'),
	Display = brain_parm.display;
else
	Display = 200;
end

[nextIX , nextDD] = vb_find_neighbor_all(Rmax, xxF, xxD, Vindx, Display);

vb_save([brain_file],'nextIX','nextDD');

proj_file_save(brain_parm);

return

function	proj_file_save(brain_parm)
% project_file save
proj_file = get_project_filename;

if isempty(proj_file)
    return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'brain_parm', brain_parm);

return
