function [Vdipole,Jdipole,err,iter] = vb_job_dipole(proj_root, dipole_parm)
% Dipole estimation by minimum error search
%  [Vdipole,Jdipole,err,iter] = vb_job_dipole(proj_root, dipole_parm)
% --- Input
% proj_root   : data directory
% dipole_parm : parameter structure
% - Input MEG file
% dipole_parm.megfile    = MEG file    (relative path from proj_root)
% - Initial dipole position
% dipole_parm.Vinit      = Initial dipole position [Ndipole x 3]
%  number of dipoles in minimum error search is given by the size of Vinit
% - Error function
% dipole_parm.errfunc    = Error function mode for dipole search
%   = 0 : 'vb_dipole_error':         for external current (Biot-Savart Eq.)
%   = 1 : 'vb_dipole_error_sarvas':  for cortical current (Sarvas Eq.)
%   = 2 : 'vb_dipole_error_magdipole':  for coil current (magnetic dipole)
% - Optional parameter
% dipole_parm.dipolefile = Output file (relative path from proj_root)
%  if this field is not exist or empty, no output file is produced.
% dipole_parm.twin_meg   =  Time window for analysis
%  if this field is not exist or empty, whole time period is used.
% dipole_parm.trial_id   =  trial number to estimate
%  if this field is not exist or empty, all trials are averaged.
% - Optimization setting (Optional)
% dipole_parm.MaxIter = 5000;	% max iteration number in optimization
% dipole_parm.TolX    = 1.e-8;	% Xνλ
% dipole_parm.TolFun  = 1.e-10;	% ؿͤνλ
% dipole_parm.mode    = 1;
%	 = 0;	% Use 'fminsearch' in Matlab
%	 = 1;	% Use 'fminunc' in Optimization Toolbox
%	 = 2;	% Use 'fminunc1'
% --- Output
%  Vdipole   : Dipole position  [Ndipole x 3]
%  Jdipole   : dipole current   [Ndipole x 3 x Time]
%  Jdipole(n ,k ,t) =  current at n-th dipole ,time 't' and k-th direction
% err  : normalized error
% iter : iteration number
%
% 2007-7-6 made by M.Sato
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

proj_root = vb_rm_trailing_slash(proj_root);

% --- Input file
megfile = [proj_root '/' dipole_parm.megfile];

% Error function name for dipole search
% err_mode 
%    = 0 : 'vb_dipole_error':         for external current (Biot-Savart Eq.)
%    = 1 : 'vb_dipole_error_sarvas':  for cortical current (Sarvas Eq.)
%    = 2 : 'vb_dipole_error_magdipole':  for external current (magnetic dipole)
err_mode = dipole_parm.errfunc;

switch	err_mode,
case	0
	errfunc = 'vb_dipole_error';
case	1
	errfunc = 'vb_dipole_error_sarvas';
case	2
	errfunc = 'vb_dipole_error_magdipole';
otherwise
	error('error function mode is incorrect')
end

% Vinit : initial dipole position (Ndipole x 3)
Vinit = dipole_parm.Vinit;

% Load MEG data
bexp = vb_load_meg_data(megfile);
[N, T, M] = size(bexp);

% set time window & trial number to estimate
if isfield(dipole_parm,'trial_id') & ~isempty(dipole_parm.trial_id)
	trial_id = dipole_parm.trial_id;
else
	trial_id = 1:M;
end

if isfield(dipole_parm,'twin_meg') & ~isempty(dipole_parm.twin_meg)
	t_begin  = dipole_parm.twin_meg(1);
	t_end    = dipole_parm.twin_meg(2);
else
	t_begin  = 1;
	t_end    = T;
end

% MEG data
bexp = bexp(:, t_begin:t_end, trial_id);
bexp = sum(bexp,3)/size(bexp,3);

% Load sensor info
[pick,Qpick,Wsensor,V0] = vb_load_sensor(megfile);

% Change center of coordinate
pick = [pick(:,1)-V0(1), pick(:,2)-V0(2), pick(:,3)-V0(3)];

% Optimization method
if isfield(dipole_parm,'mode') & ~isempty(dipole_parm.mode)
	opt_mode = dipole_parm.mode;
else
	opt_mode = vb_tool_check('Optimization');
	%	 = 0;	% Use 'fminsearch'
	%	 = 1;	% Use 'fminunc' in Optimization Toolbox
end

% Dipole position search
[Vdipole, err, iter] = ...
  vb_dipole_search(Vinit,bexp,pick,Qpick,Wsensor,errfunc,dipole_parm,opt_mode);

% Dipole current time cource for fixed position dipole
[Jdipole, err] = ...
  vb_dipole_current(Vdipole, bexp, pick, Qpick, Wsensor, [0 0 0], err_mode);

% Right-hand SPM (m) coordinate
Vdipole = [ Vdipole(:,1)+V0(1), Vdipole(:,2)+V0(2), Vdipole(:,3)+V0(3)];

% Save result
% --- Output file
if isfield(dipole_parm,'dipolefile') & ~isempty(dipole_parm.dipolefile)
	dipolefile = [proj_root '/' dipole_parm.dipolefile];
	vb_save(dipolefile, 'Vdipole','Jdipole','dipole_parm','err');
end
