function [V,F,xx,BV_index,Vinfo] = vb_job_brain_subj(proj_root,brain_parm, mode)
% Make whole brain data (cortex sheet) by reducing BV/FS surface data.
% (VBMEG public function)
%
% [syntax]
% vb_job_brain_subj(proj_root,brain_parm);
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% brain_parm: <<struct>> Parameters for creating cortical surface model
% --- fields of brain_parm
% analyze_file: <<string>> MRI filename (.nii/.hdr)
% brain_parm.FS_left_file         = [fs_dir '/bem/lh.smoothwm.asc'];
% brain_parm.FS_right_file        = [fs_dir '/bem/rh.smoothwm.asc'];
% brain_parm.FS_left_infl_file    = [fs_dir '/bem/lh.inflated.asc'];
% brain_parm.FS_right_infl_file   = [fs_dir '/bem/rh.inflated.asc'];
% brain_parm.FS_left_curv_file    = [fs_dir '/bem/lh.curv.asc'];
% brain_parm.FS_right_curv_file   = [fs_dir '/bem/rh.curv.asc'];
% brain_parm.FS_left_label_file  = [fs_dir '/label/lh.cortex.label'];
% brain_parm.FS_right_label_file = [fs_dir '/label/rh.cortex.label'];
% 
% brain_parm.FS_left_sphere_file  = [fs_dir,  'lh.sphere.reg.asc'];
% brain_parm.FS_right_sphere_file = [fs_dir,  'rh.sphere.reg.asc'];
% brain_parm.FS_sphere_key        = 'sphere.reg';
%
%  brain_file  : <<string>> Cortical surface model file (.brain.mat).
%  act_file    : <<string>> Cortical activity map file (.act.mat).
%  area_file   : <<string>> Cortical area file (.area.mat).
%  Nvertex     : <<int>>    Total number of vertices for cortical surface model
%                           Priority: Nvertex > reduce_ratio
%  Rmax        : <<float>>  Maximum radius for neighbor search 
%  N_step      : <<int>>    Division number of z axis in matching between
%                           cortical surface model and MRI points
%  display     : <<int>>    Display progress for this steps
%  
%  (parameters for BrainVoyager cortical surface model) <<string>>
%  BV_left_file      : GM/WM boundary file (.srf) for left hemisphere 
%  BV_right_file     : GM/WM boundary file (.srf) for right hemisphere 
%  
% ---
% 
% [output]
% The following files are created: 
%  VBMEG cortical surface model file (.brain.mat)
%  VBMEG area file (.area.mat)
%  VBMEG activity map file (.act.mat)
%
% The following variables are stored in .brain.mat file: 
% V         : Cortical vertex point cordinate (SPM_Right_m) 
% F         : Patch index structure
% xx        : Normal vector to cortical surface 
% xxA       : area asigned for each vertex ([m^2])
%
% BV_index  : original vertex index in BV/FS corresponding to brain
% Vinfo     : Vertex dimension structure
%
% xxF{i}    : Next-neighbor index for the vertex-i
% xxD{i}    : Distance between next-neighbor and the vertex-i
% xxT{n}    : vertex index for n-th triangle
% nextIX{i} : Neighbor index list for the vertex-i
% nextDD{i} : Distance from the vertex-i
% 
% The following areas are automatically created and stored in .area.mat
% file: 
% areakey = 'Cortex'         : whole brain
% areakey = 'cortex_region'  : cortex region without corpus area
% areakey = 'corpus_area'    : corpus area
%
% [history]
% Line 117 was modified by Kawawaki, Dai at 06/04/07
% 2006/07/21 M. Sato 
%  Make reduced cortex for soft normal constraint
% 2007/03/05 O. Yamashita
%  Add lines to compute MNI and Talairach coordinate and two area files
%  derived from anatomical divisions. These lines are only valid when 
%  spm_normalization_file is specified.
% 2008-12-24 Taku Yoshioka
%  Reduce model supression mode
%  vb_disp() is used for displaying message
%  Show figures for checking standard brain mask
% 2010-05-28 Taku Yoshioka
%  Minor change
% 2011-01-28 rhayashi
%  call vb_job_inflate() when brain_parm have fields for inflate model.
% 2017-03-14 rhayashi
%  exclude corpus area from 'Cortex'. 
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

% Set BV-SPM version (mode = 0 : SBI-compatible)
if ~exist('mode','var'), mode = 1; end;

proj_root = vb_rm_trailing_slash(proj_root);

%
% Do not modify following lines
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

% Output file path
brain_file = [proj_root filesep brain_parm.brain_file];
area_file  = [proj_root filesep brain_parm.area_file ];
act_file   = [proj_root filesep brain_parm.act_file  ];

% ID for MRI image data
[udir, fname] = vb_get_file_parts(brain_parm.analyze_file);
MRI_ID = fname;

%-------------------------------------------------------
% Make cortex coordinate from Brain-Voyager srf-file
%-------------------------------------------------------
switch	mode
case	0
	% BV-Vox-SPM version (SBI-compatible)
	[V,F,xx,BV_index,Vinfo] = vb_make_brain_data(brain_parm);
	DEBUG_MODE = 0;
case	1
	% BV-SPM version
	[V,F,xx,BV_index,Vinfo] = vb_make_brain_data2(brain_parm);
	DEBUG_MODE = 0;
case	2
	% BV-Vox-SPM version (SBI-compatible)
	[V,F,xx,BV_index,Vinfo] = vb_make_brain_data(brain_parm);
	DEBUG_MODE = 2;
case	3
	% BV-SPM version
	[V,F,xx,BV_index,Vinfo] = vb_make_brain_data2(brain_parm);
	DEBUG_MODE = 2;
end

if DEBUG_MODE==2, return; end;

%----------------------------------------
% Search next-point index and distance
%----------------------------------------
tic;
vb_disp_nonl('Search next-point index and distance ', ...
        VERBOSE_LEVEL_NOTICE);

[xxD, xxF, xxT] = vb_next_distance( F.F3, V ); % unit : m

vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

% total number of vertex
Ndipole = Vinfo.Ndipole;
Vindx   = 1:Ndipole;

%---------------------------------------
% Smoothing normal vectors (Double mean)
%---------------------------------------
tic
vb_disp_nonl('Mean of patch normal ', VERBOSE_LEVEL_NOTICE);

% Mean of normal vector for neighboring patch 
xx = vb_mean_normal_vector_1(Vindx ,F,V,xx);
% Mean of normal vector for neighbor vertex
xx = vb_mean_normal_vector_4(Vindx ,F,V,xx,xxF);

vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

%----------------------------------------------
% Calculate area assigned to the vertex
%----------------------------------------------
tic; 
[xxA] = vb_calc_patch_area(V, F.F3, xxT);
vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

%
% Save cortical surface model
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp(['Save brain model '], VERBOSE_LEVEL_NOTICE);
vb_disp(sprintf('     filename = %s', brain_file), ...
        VERBOSE_LEVEL_NOTICE);
vb_save([brain_file],...
     'F','V','xx','xxF','xxD','xxA','Vinfo','BV_index','MRI_ID');

if DEBUG_MODE==1, return; end;

%----------------------------------------------
% Search neighbor points along cortex sheet
%----------------------------------------------
Rmax    = brain_parm.R_max; 
Display = brain_parm.display;

[nextIX , nextDD] = vb_find_neighbor_all(Rmax, xxF, xxD, Vindx, Display);

vb_disp('Save neighbor index of cortical vertex', ...
        VERBOSE_LEVEL_NOTICE);
vb_save([brain_file],'nextIX','nextDD');

%--------------------------------------------------------------------------
% Calculate MNI coordinate and Talairach coordinate and
% two area files derived from two anatomical divisons (added by OY 07/03/05)
%--------------------------------------------------------------------------
mni_calculation = ...
    (isfield(brain_parm, 'spm_normalization_file') ...
      && ~isempty(brain_parm.spm_normalization_file)) ...
   ||(isfield(brain_parm, 'FS_left_sphere_file') ... 
      && ~isempty(brain_parm.FS_left_sphere_file) && ...
    isfield(brain_parm, 'FS_right_sphere_file') ...
      && ~isempty(brain_parm.FS_right_sphere_file));

if mni_calculation
  atlas_dir = 'MNI_atlas_templates/';
  EXT_brain = '.brain.mat';
  brain_id       = brain_file(1:findstr(brain_file,EXT_brain)-1);
    
  %% MNI coordinate
  tic 
    % Computing MNI and Talairach coordinate
    vb_job_brain_add_mni_tal_coord(proj_root, brain_parm, false);

    %% AAL atlas
    tic 
    mniparm.brainfile  = brain_file;
    vb_disp_nonl(['Creating an area file consisting of AAL anatomical' ...
             'division: '], VERBOSE_LEVEL_NOTICE);
    atlas_id = 'aal';
    
    mniparm.atlas_text = [atlas_dir atlas_id '.txt'];
    mniparm.atlas_file = [atlas_dir atlas_id '.img'];
    mniparm.atlas_id   = atlas_id;
    mniparm.atlas_dir  = atlas_dir;


    mniparm.save_areafile  = [brain_id  '_'  atlas_id  '.area.mat'];
    mniparm.save_atlasfile = [brain_id  '_atlas.act.mat'];	% Atlas label

    vb_atlas2vb_aal(mniparm)
    
    vb_disp_nonl(sprintf('Done...%f[sec]\n\n',toc));   

    
    %% Brodmann atlas
    tic 
    vb_disp_nonl(['Creating an area file consisting of Brodmann ' ...
             'anatomical division: ']);
    
    atlas_id = 'brodmann';
    
    mniparm.atlas_text = [atlas_dir atlas_id '.txt'];
    mniparm.atlas_file = [atlas_dir atlas_id '.img'];
    mniparm.atlas_id   = atlas_id;

    mniparm.save_areafile  = [brain_id  '_'  atlas_id  '.area.mat'];

    vb_atlas2vb(mniparm)

    vb_disp_nonl(sprintf('Done...%f[sec]\n\n',toc), VERBOSE_LEVEL_NOTICE);
else 
    vb_disp('MNI and Talairach coordinate are not computed.', ...
            VERBOSE_LEVEL_NOTICE);
    vb_disp(['You can obtain MNI and Talairach coordinate by ' ...
             'running vb_atlas2vb.m later '], VERBOSE_LEVEL_NOTICE);
end

%--------------------------------------
% Make new area and act files
%--------------------------------------
act_new.key     = 'Uniform';
act_new.xxP     = ones(Ndipole,1);
act_new.comment = 'homogeneous fMRI constraint';

vb_add_act([act_file], act_new, MRI_ID, OFF);

% Make Area file
Vindx  = [Vinfo.cortexL; Vinfo.cortexR];
AreaNew.key      = 'Cortex';
AreaNew.Iextract = Vindx; % rhayashi 2017/3/14

vb_add_area([area_file], AreaNew, MRI_ID, OFF);

%---------------------------------------------------
% Find original vertex near the cortical vertex
%---------------------------------------------------
normal_stat = vb_original_normal_statics(proj_root,brain_parm, BV_index);

vb_disp('Save neighbor statics in the original brain', ...
        VERBOSE_LEVEL_NOTICE);
vb_save(brain_file,'normal_stat');

%--------------------------
% Make reduced cortex
%--------------------------
if isfield(brain_parm,'make_reduce') & brain_parm.make_reduce, 
  vb_job_reduced_cortex(proj_root,brain_parm);
end

%--------------------------
% Import inflate model
%--------------------------
if has_inflate_model(brain_parm)
  vb_job_inflate(proj_root, brain_parm);
end

%-----------------------------------------
% Check brain model by 3D & MRI image
%-----------------------------------------
vb_check_brain_model(proj_root,brain_parm);

%--------------------------
% Save project file
%--------------------------
proj_file = get_project_filename;
if ~isempty(proj_file)
    project_file_mgr('load', proj_file);
    project_file_mgr('add', 'brain_parm', brain_parm, 'vb_job_brain');
end

% Plot MNI calculation result
% if mni_calculation
%     try
%         vb_check_mni_coregistration(brain_file);
%     catch
%         vb_disp(['Failed to plot MNI coordinate value(Out of Memory)', ...
%                  'try again: vb_check_mni_coregistration(brain_file)'], ....
%                  'WARNING');
%         
%     end
% end

return;

function [result] = has_inflate_model(brain_parm)
% Check if brain_parm contains valid fields for inflate model.
% result : return bit pattern
%          = 01(1 as int): BV inflate data is set.
%          = 10(2 as int): FS inflate data is set.
%          = 11(3 as int): Both inflate model were set
result = 0;
BV_BIT = 1; FS_BIT = 2;
% bit all on
result = bitset(result, BV_BIT, 1);
result = bitset(result, FS_BIT, 1);

BV_fields = {'BV_left_infl_file', 'BV_right_infl_file'};
FS_fields = {'FS_left_infl_file',  'FS_right_infl_file', ...
             'FS_left_curv_file', 'FS_right_curv_file'};
for k=1:length(BV_fields)
  if ~isfield(brain_parm, BV_fields{k}) || ...
     exist(brain_parm.(BV_fields{k}), 'file') ~= 2
    % bit off 
    result = bitset(result, BV_BIT, 0);
    break;
  end
end
for k=1:length(FS_fields)
  if ~isfield(brain_parm, FS_fields{k}) || ...
     exist(brain_parm.(FS_fields{k}), 'file') ~= 2
    % bit off 
    result = bitset(result, FS_BIT, 0);
    break;
  end
end
