function	[V,F,xx,brain_parm] = vb_job_brain_std2sbj(proj_root,brain_parm)
% Make smooth cortex from standard brain mask
%  [V,F,xx,brain_parm] = vb_job_brain_std2sbj(proj_root,brain_parm)
% --- Input 
% proj_root  : root directory for data file
% brain_parm : parameter structure
% - Input file
% brain_parm.subj_mask   : Subject brain mask file
% brain_parm.analyze_file: analyze_file
% - Output file
% brain_parm.brain_file  : brain_file 
% brain_parm.area_file   : area_file  
% brain_parm.act_file    : act_file   
% - Parameter
% brain_parm.Nvertex = 3000;  number of cortical vertex
% brain_parm.Radius = [2 -2]; morphological smoothing
% brain_parm.gray_val : threshold value for gray matter
%   if it is empty, only brain mask is used
%   if is is given, voxel with larger intensity than 'gray_val' is selected
%
% ---- Output (also saved in the brain_file)
% V      : Cortical vertex point cordinate (SPM_Right_m) 
% F      : Patch index structure
% xx     : Normal vector to cortical surface 
% ---- Saved variable in the brain_file
% xxA       : area for vertex
%
% xxF{i}    : Next-neighbor index for the vertex-i
% xxD{i}    : Distance between next-neighbor and the vertex-i
% xxT{n}    : vertex index for n-th triangle
% nextIX{i} : Neighbor index list for the vertex-i
% nextDD{i} : Distance from the vertex-i
%
% Vinfo     : Vertex dimension structure
%
% M. Sato 2007/7/05
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

DEBUG_MODE=0;

proj_root = vb_rm_trailing_slash(proj_root);
subj_mask    = [proj_root filesep brain_parm.subj_mask];
analyze_file = brain_parm.analyze_file;

% Output file path
brain_file = [proj_root filesep brain_parm.brain_file];
area_file  = [proj_root filesep brain_parm.area_file ];
act_file   = [proj_root filesep brain_parm.act_file  ];

[V, F, xx, Xcenter] = vb_make_smooth_cortex_surf(subj_mask,analyze_file,brain_parm);

msg = {'Cortical surface is plotted'; 'Is this OK ?'};
str = questdlg(msg,'Check result','Yes', 'No','Yes');
if strcmp(str,'No'), return; end;

%%%% DEBUG %%%%
if DEBUG_MODE==2, return; end;

% ID for MRI image data
[udir, fname] = fileparts(analyze_file);
MRI_ID = fname;

[V,F,xx,Vinfo] = vb_get_cortex_info(V, F, xx, Xcenter);

%
% --- Search next-point index and distance
%
tic;
fprintf('--- Search next-point index and distance ');

[xxD, xxF, xxT] = vb_next_distance( [F.F3L ; F.F3R], V ); % unit : m

fprintf('%f[sec]\n',toc);

%
% --- Calculate area assigned to the vertex
%
tic

[xxA] = vb_calc_patch_area(V, F.F3, xxT);

fprintf('%f[sec]\n',toc);

fprintf(['--- Save brain model \n']);
fprintf('     filename = %s\n', brain_file );

vb_save([brain_file], 'F','V','xx','xxF','xxD','xxA',...
     'Vinfo','MRI_ID');

% total number of vertex
Ndipole = Vinfo.Ndipole;
Vindx   = 1:Ndipole;

% Make activity map file
act_new.key     = 'Uniform';
act_new.xxP     = ones(Ndipole,1);
act_new.comment = 'artificial data';

vb_add_act([act_file], act_new, MRI_ID, OFF);

% Make Area file
AreaNew.key      = 'Cortex';
AreaNew.Iextract = [1:Ndipole]'; 

vb_add_area([area_file], AreaNew, MRI_ID, OFF);

if DEBUG_MODE==1, 
	figure;
	hist(xxA*1e6, 0:0.25:10);
	xlim([0 10])
	title('Histgram of patch area')
	xlabel('Area')
	return; 
end;
%
% Search neighbor points along cortex sheet
%

if isfield(brain_parm,'R_max'),
	Rmax    = brain_parm.R_max; 
else
	Rmax    = 18e-3; 
end
if isfield(brain_parm,'R_max'),
	Display = brain_parm.display;
else
	Display = 200;
end

[nextIX , nextDD] = vb_find_neighbor_all(Rmax, xxF, xxD, Vindx, Display);

vb_save([brain_file],'nextIX','nextDD');


% Check brain model by 3D & MRI image
vb_check_brain_model(proj_root,brain_parm);

figure;
hist(vb_cell_merge(xxD)*1000,100);
title('Histgram of Vertex distance [mm]')

proj_file_save(brain_parm);

return

function	proj_file_save(brain_parm)
% project_file save
proj_file = get_project_filename;

if isempty(proj_file)
    return;
end

project_file_mgr('load', proj_file);
project_file_mgr('add', 'brain_parm', brain_parm);

return
