function job_view_leadfield(basisfile,brainfile,model)
% View spatial pattern of lead field in sensor space
%
% --- Syntax
% job_view_leadfield(basisfile,brainfile,model)
%
% --- Input
% basisfile: Leadfield file (.basis.mat)
% brainfile: Cortex model file (.brain.mat)
% model    : 0 for original (folded), 1 for inflated display
%
% --- Example
% job_view_leadfield('./sbj1/basis/meg_visual_exp.basis.mat', ...
%                    './sbj1/data/3D.brain.mat');
%
% --- Variables in vbmeg_inst.viewlf{n}
% .brainfile
% .model (indicating original or inflated display)
% .V (Vertex coordinate values for original cortical surface model)
% .Vinf (Vertex coordinate values for inflated cortical surface model)
% .F (Face information)
% .inf_C (Face color for inflated model)
% .xxA (Cortical area size for vertices)
% .Vtal (Standard brain coordinate values)
% .nextDD (Distance to the neighbouring vertices)
% .nextIX (Indices of the neighbouring vertices)
% .J (Activity pattern; lf norm, correlation)
% .plot_parm (Structure of plot parameters)
%
% .basisfile
% .basis_org (original lead field pattern)
%
% --- History
% 2008-01-26 Taku Yoshioka
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% Input parameter check
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if nargin<3, model = 1; end
if nargin<2, brainfile = []; end
if nargin<1, basisfile = []; end

inst_id = create_instance;
if isempty(inst_id), return; end; 

%
% Create GUI and update plots
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
create_GUI(inst_id,basisfile,brainfile,model);
job_view_lf_update_cortexpattern(inst_id);
job_view_lf_update_sensorpattern(inst_id);

%%%
%%% Inner functions 
%%%

%
% Create instance of GUI
% i: Instance ID of job_plot_currentmap
%
function inst_id = create_instance

global vbmeg_inst

if isfield(vbmeg_inst,'viewlf'),
  for inst_id=1:vbmeg_inst.const.VIEWLF_MAX
    if isempty(vbmeg_inst.viewlf{inst_id}), return; end;
  end
  fprintf(['The number of instances of ''job_view_leadfield''' ...
	   ' GUIs reaches the maximum number.\n']);
  inst_id = [];
elseif isfield(vbmeg_inst,'const'), 
  vbmeg_inst.viewlf = cell(vbmeg_inst.const.VIEWLF_MAX,1);
  inst_id = 1;
else
  fprintf(['''vbmeg_inst'' was not correct. You have to invoke' ...
	   '''vbmeg'' to create an instance of VBMEG.\n']); 
  inst_id = []; 
end

%
% Create GUI
%
function create_GUI(inst_id,basisfile,brainfile,model)

global vbmeg_inst;

%
% Initialize parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Cortical surface model
viewlf.brainfile = brainfile;
viewlf.V = [];
viewlf.Vinf = [];
viewlf.F = [];
viewlf.inf_C = [];
viewlf.xxA = [];
viewlf.nextDD = [];
viewlf.nextIX = [];
%viewlf.Vtal = [];
%viewlf.J = [];

% Lead field and sensor data
viewlf.basisfile = [];
viewlf.basis_org = [];
viewlf.basis_org_smoothed = [];
viewlf.pick = [];
viewlf.MEGinfo = [];
  
% Display parameters
viewlf.ix_selected_vertex = 1;
viewlf.Jmin_plot = -1;
viewlf.Jmax_plot = 1;
viewlf.Bmin_plot = -100;
viewlf.Bmax_plot = 100;
viewlf.sensor_type = 2;
viewlf.plot_parm = vb_set_plot_parm;
viewlf.plot_parm.index = [];
viewlf.plot_parm.LRflag = 'LR';
%viewlf.ix_selected_area = [];

%
% GUI open
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h_fig = open('job_view_leadfield.fig');
set(h_fig,'HandleVisibility','on');
drawnow; 
set(h_fig,'Pointer','watch');

% Input parameter
viewlf.model_type = model;

%
% Handles of GUI components
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Figure and axis
viewlf.h_fig = h_fig;
viewlf.h_cortex = get_child_handle(h_fig,'plot_cortex');
%viewlf.h_cbar_cortex = get_child_handle(h_fig,'cortex_colorbar');
viewlf.h_sensor = get_child_handle(h_fig,'plot_sensor');
%viewlf.h_cbar_sensor = get_child_handle(h_fig,'sensor_colorbar');

% Editable text
viewlf.ed1 = get_child_handle(h_fig,'vertex_index');
viewlf.ed2 = get_child_handle(h_fig,'lf_power');
viewlf.ed3 = get_child_handle(h_fig,'filter_radius');

% Push button
viewlf.pb1 = get_child_handle(h_fig,'left_button');
viewlf.pb2 = get_child_handle(h_fig,'right_button');
viewlf.pb3 = get_child_handle(h_fig,'top_button');
viewlf.pb4 = get_child_handle(h_fig,'bottom_button');
viewlf.pb5 = get_child_handle(h_fig,'rotate');
viewlf.pb6 = get_child_handle(h_fig,'select_vertex');
viewlf.pb7 = get_child_handle(h_fig,'print');

% Radio button
viewlf.rb1 = get_child_handle(h_fig,'model_type_original_rb');
viewlf.rb2 = get_child_handle(h_fig,'model_type_inflate_rb');

% Checkbox
viewlf.cb1 = get_child_handle(h_fig,'left_brain');
viewlf.cb2 = get_child_handle(h_fig,'right_brain');
viewlf.cb3 = get_child_handle(h_fig,'fix_cortex_colorbar');
viewlf.cb4 = get_child_handle(h_fig,'fix_sensor_colorbar');

% Menu
viewlf.load_cortex_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'load_cortex_menu');
viewlf.load_leadfield_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'),...
                       'load_leadfield_menu');
viewlf.loaded_files_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'loaded_files_menu');
viewlf.close_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'close_menu');

% Set GUI values
set(viewlf.ed1,'String','1');

%
% Callback functions
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
inst_str = num2str(inst_id); 
viewlf_str = ['vbmeg_inst.viewlf{' inst_str '}'];

% Remove instance after window close
command = ['global vbmeg_inst; '...
	   'vbmeg_inst.viewlf{' inst_str '}=[];'];
set(h_fig,'DeleteFcn',command);

% Change vertex 
command = ['global vbmeg_inst;' ...
	   viewlf_str '.ix_selected_vertex = ' ...
	   'str2num(get(' viewlf_str '.ed1,''String''));' ...
           'job_view_lf_update_cortexpattern(' inst_str ');' ...
	   'job_view_lf_update_sensorpattern(' inst_str ');'];
set(viewlf.ed1,'Callback',command);

% Change filter radius
command = ['job_view_lf_change_filter_radius(' inst_str ');'];
set(viewlf.ed3,'Callback',command);

% Change angle
command = ['job_view_lf_change_angle(' inst_str ',''left'');'];
set(viewlf.pb1,'Callback',command);
command = ['job_view_lf_change_angle(' inst_str ',''right'');'];
set(viewlf.pb2,'Callback',command);
command = ['job_view_lf_change_angle(' inst_str ',''top'');'];
set(viewlf.pb3,'Callback',command);
command = ['job_view_lf_change_angle(' inst_str ',''bottom'');'];
set(viewlf.pb4,'Callback',command);

% Rotation of cortex model
command = ['job_view_lf_push_rotate_button(' inst_str ');']; 
set(viewlf.pb5,'Callback',command);

% Select vertex
command = ['job_view_lf_selvertex(' inst_str ');'];
set(viewlf.pb6,'Callback',command);

% Print figure (not implemented yet)
command = ['job_view_lf_print(' inst_str ');'];
set(viewlf.pb7,'Callback',command);

% Change filter radius
%command = ['job_plot_cmap_update_filterradius(' inst_str ');' ...
%           'job_plot_cmap_update_spatialpattern(' inst_str ');' ...
%	   'job_plot_cmap_update_timecourse(' inst_str ');'];
%set(viewlf.ed5,'Callback',command);

% Folded/Inflated model
%model_type_group = [viewlf.model_type_inflate_rb,
%                    viewlf.model_type_original_rb];
model_type_group = [viewlf.rb1, viewlf.rb2];
command = ['job_view_lf_change_modeltype(' inst_str ');'];
set(model_type_group, 'Callback', command);
set(model_type_group, 'Value',get(viewlf.rb1,'Min')); % both off
if model == 0, set(viewlf.rb1,'Value',get(viewlf.rb1,'Max'));
else set(viewlf.rb2,'Value',get(viewlf.rb2,'Max')); end

% Left/Right brain
command = ['job_view_lf_update_cortexpattern(' inst_str ');'];
set(viewlf.cb1,'Callback',command);
set(viewlf.cb2,'Callback',command);

% Fix color scale
command = ['job_view_lf_update_cortexpattern(' inst_str ');'];
set(viewlf.cb3,'Callback',command);
command = ['job_view_lf_update_sensorpattern(' inst_str ');'];
set(viewlf.cb4,'Callback',command);

% Cancel
set(h_fig,'WindowButtonUpFcn','rotate3d off;');

% Load cortical surface model
command = ['[brain_dir,brain_file] = vb_file_select' ...
           '({''.brain.mat''},''Load cortical surface model'');' ...
           'if ~isempty(brain_file), ' ...
	   'job_view_lf_load_cortex(' inst_str ',' ...
           '[brain_dir filesep brain_file{1}]); end'];
set(viewlf.load_cortex_menu, 'Callback', command);

% Load lead field data
command = ['[basis_dir,basis_file] = vb_file_select' ...
           '({''.basis.mat''},''Load leadfield data'');' ...
           'if ~isempty(basis_file), ' ...
	   'job_view_lf_load_leadfield(' inst_str ',' ...
           '[basis_dir filesep basis_file{1}]); end'];
set(viewlf.load_leadfield_menu, 'Callback', command);

% Show loaded files (not yet implemented)
command = ['job_view_lf_show_loaded_files(' inst_str ');']; 
set(viewlf.loaded_files_menu, 'Callback', command);

% Close window
command = 'close;';
set(viewlf.close_menu,'Callback',command);

%
% Properties of axis for cortex plot
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
set(h_fig,'Renderer','zbuffer');
set(h_fig,'CurrentAxes',viewlf.h_cortex);
axis off;
set(h_fig,'CurrentAxes',viewlf.h_sensor);
axis off;
%axis tight; % 2006/11/22 hayashi delete
%axis equal; % These commands causes axes LimMode change(auto-->manual)
             % after MATLAB 7.0.1. and brain model is not displayed. 

%
% Set instance
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vbmeg_inst.viewlf{inst_id} = viewlf; 

% Set pointer to arrow
set(h_fig,'Pointer','arrow');

%
% Load cortical surface model
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(brainfile),
  job_view_lf_load_cortex(inst_id,brainfile);
end

%
% Load estimated current
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(basisfile),
  job_view_lf_load_leadfield(inst_id,basisfile);
end
