function [ix_time_from, ix_time_to, cancelled] = ...
                select_time_window_gui(varargin)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

key = varargin{1};

switch(key)

    case 'init'
        fig = openfig(mfilename);
        H = guihandles(fig);
        % Time window parameters
        parm = struct;
        parm.ix_origin = varargin{2};
        parm.ix_range_from = varargin{3};
        parm.ix_range_to = varargin{4};
        parm.sampling_freq = varargin{5};
        parm.ix_initial_from = varargin{6};
        parm.ix_initial_to = varargin{7};

        % save GUI component handles
        setappdata(fig, 'H', H);

        % save time window parameters
        set_time_window_parameter(parm);

        % GUI close by CANCEL button or 'x' button.
        setappdata(fig, 'cancelled', true);
        
        % Initialize screen display.
        init_display(H);

        % into blocking till figure property visible off
        waitfor(fig, 'Visible', 'off');

        % get output values
        cancelled = getappdata(fig, 'cancelled');
        if cancelled
            ix_time_from = 0;
            ix_time_to = 0;
        else
            [parm] = get_time_window_parameter;
            ix_time_from = parm.ix_time_from;
            ix_time_to   = parm.ix_time_to;
        end

        % delete figure
        clear_time_window_parameter;
        delete(fig);

    case 'exit'
        setappdata(gcf, 'cancelled', true);
        % exit blocking
        set(gcf, 'Visible', 'off');

    case 'callback'
       H = getappdata(gcf, 'H');
       hObj = varargin{2};
       callback(H, hObj);
end

function callback(H, hObj)

switch(hObj)
    case H.ok_push
        [ix_time_from, ix_time_to, err] = get_time_window_from_gui(H);
        if err || (ix_time_from > ix_time_to)
            errordlg('Time window is invalid.', 'error');
            return;
        else
            % set return parameter
            parm = get_time_window_parameter;
            parm.ix_time_from = ix_time_from;
            parm.ix_time_to   = ix_time_to;
            set_time_window_parameter(parm);

            % exit blocking
            setappdata(gcf, 'cancelled', false);
            set(gcf, 'Visible', 'off');
        end
    case H.cancel_push
        % exit blocking
        setappdata(gcf, 'cancelled', true);
        set(gcf, 'Visible', 'off'); 

    case { H.time_window_from_edit, H.time_window_to_edit }
        % get input
        editbox_string = get(hObj, 'String');
        time_window_msec = str2double(editbox_string);
        
        % convert time to index
        [index, err] = vb_time_to_index(time_window_msec);
        if err
            if hObj == H.time_window_from_edit
                rollback_time_window('start', H);
            elseif hObj == H.time_window_to_edit
                rollback_time_window('end', H);
            end
            errordlg('Invalid time was specified.', 'error');
            return;
        end
        
        % convert index to time
        [time, err]  = vb_index_to_time(index);
        if err
            if hObj == H.time_window_from_edit
                rollback_time_window('start', H);
            elseif hObj == H.time_window_to_edit
                rollback_time_window('end', H);
            end
            errordlg('Invalid time was specified.', 'error');
            return;
        end

        % set Time window
        if hObj == H.time_window_from_edit
            set_time_window('start', time, H);
        elseif hObj == H.time_window_to_edit
            set_time_window('end', time, H);
        end
end

function init_display(H)

    % get window parameter
    parm = get_time_window_parameter;
    
    %
    % --- Set Range
    %
    time = vb_index_to_time(parm.ix_range_from);
    set(H.range_from_edit, 'String', sprintf('%.2f', time));
    time = vb_index_to_time(parm.ix_range_to);
    set(H.range_to_edit, 'String', sprintf('%.2f', time));

    %
    % --- Set initial Time window
    time = vb_index_to_time(parm.ix_initial_from);
    set_time_window('start', time, H);
    time = vb_index_to_time(parm.ix_initial_to);
    set_time_window('end', time, H);
    
function set_time_window(key, time, H)
% This function sets time window to GUI and data store.
% [IN]
%    key  : Time window type.  'start' or 'end'
%    time : Time(msec)
%    H    : GUI handles
%
    % ӻ꤬ФΤǡǥåȥܥåɽƤͤϻѤʤ
    % index_to_timeѴ줿ͤfigure¸
    %      : setappdata(H.fig, 'time_window_xxx_edit', time)
    switch(key)
        case 'start'
            set(H.time_window_from_edit, 'String', sprintf('%.2f', time));
            setappdata(H.fig, 'time_window_from_edit', time);
        case 'end'
            set(H.time_window_to_edit, 'String', sprintf('%.2f', time));
            setappdata(H.fig, 'time_window_to_edit', time);
    end

function rollback_time_window(key, H)
% This function rollback time window GUI.
%
% [IN]
%    key  : Time window type.  'start' or 'end'
%    H    : GUI handles
%
    switch(key)
        case 'start'
            time = getappdata(H.fig, 'time_window_from_edit');
            set_time_window('start', time, H);
        case 'end'
            time = getappdata(H.fig, 'time_window_to_edit');
            set_time_window('end', time, H);
    end
            
function [ix_time_from, ix_time_to, err] = get_time_window_from_gui(H)
    err = false;
    ix_time_from = 0;
    ix_time_to = 0;

    time_window_from_msec = getappdata(gcf, 'time_window_from_edit');
    if isempty(time_window_from_msec)
        err = true;
        return;
    end
    time_window_to_msec   = getappdata(gcf, 'time_window_to_edit');
    if isempty(time_window_to_msec)
        err = true;
        return;
    end
    
    [ix_time_from, err] = vb_time_to_index(time_window_from_msec);
    if err, return; end
    [ix_time_to,   err] = vb_time_to_index(time_window_to_msec);
    if err, return; end

function [meg_index, err] = vb_time_to_index(time)
% This function returns nearest rounded up sampling index from
% specified time.
% [IN]
%      time       : time point(msec)
%      SampleFreq : Sampling frequency
%
% [OUT]
%      meg_index : nearest rounded up sampling index from specified time.
%      err       : true or false(true means something error occured.)

    err = false;
    meg_index = [];

    parm = get_time_window_parameter;
    SampleFreq = parm.sampling_freq;
    ix_origin     = parm.ix_origin;
    ix_range_from = parm.ix_range_from;
    ix_range_to   = parm.ix_range_to;

    try
        %
        % --- make table (meg_index_list, time_list)
        %
        
        % meg index list
        meg_index_list = parm.ix_range_from:1:parm.ix_range_to;

        % sampling length(msec).
        sampling_length = 1000/SampleFreq;

        % time list of sample points(msec)
        start_point = ix_range_from - ix_origin;
        end_point = ix_range_to - ix_origin;
        time_list = sampling_length * start_point : ...
                            sampling_length : ...
                            sampling_length * end_point;

        for k=1:length(time)
            % repaired out of range input
            if time(k) < min(time_list)
                time(k) = min(time_list);
            elseif time(k) > max(time_list)
                time(k) = max(time_list);
            end

            % Index of MEG data
            table_ix = min(find(time_list>=time(k)));
            meg_index(k) = meg_index_list(table_ix);
        end
    catch
        err = true;
    end

function [time, err] = vb_index_to_time(meg_index)
% This function returns time that corresponds to sampling data index. 
% [IN]
%      meg_index : index of meg data.
% [OUT]
%      time      : time that corresponds to sampling data index. 
%      err       : true or false(true means something error occured.)

    err = false;
    time = [];

    parm = get_time_window_parameter;
    SampleFreq = parm.sampling_freq;
    ix_origin     = parm.ix_origin;
    ix_range_from = parm.ix_range_from;
    ix_range_to   = parm.ix_range_to;

    try
        %
        % --- make table (meg_index_list, time_list)
        %

        % meg index list
        meg_index_list = parm.ix_range_from:1:parm.ix_range_to;
        
        % sampling length(msec).
        sampling_length = 1000/SampleFreq;

        % time list for index points(msec)
        start_point = ix_range_from - ix_origin;
        end_point = ix_range_to - ix_origin;
        time_list = sampling_length * start_point : ...
                            sampling_length : ...
                            sampling_length * end_point;
        
       for k=1:length(meg_index)
            % array over flow check
            if meg_index(k) > ix_range_to
                meg_index(k) = ix_range_to;
            end
            if meg_index(k) < ix_range_from
                meg_index(k) = ix_range_from;
            end
            % return time
            table_ix(k) = find(meg_index_list == meg_index);
            time(k) = time_list(table_ix(k));
        end
    catch
        err = true;
        return;
    end

function set_time_window_parameter(parm)
    global select_time_window_gui_abc123;
    select_time_window_gui_abc123 = parm;
function [parm] = get_time_window_parameter;
    global select_time_window_gui_abc123;
    parm = select_time_window_gui_abc123;
function clear_time_window_parameter;
    global select_time_window_gui_abc123;
    clear select_time_window_gui_abc123;
