function [F,ix,meg_filt,Tperiod,Tnext] = inverse_filter(resultfile,ix0)
% [F,ix,meg_filt,Tperiod,Tnext] = inverse_filter(resultfile,ix0)
%
% Return inverse filters based on the parameters saved in a 
% specified result file. In current version, the number of 
% leadfield components in the inverse filter calculation must be
% one (L=1). If not so, the function will not work correctly. 
%
% 2004-09-19 Taku Yoshioka
%
% --- Input parameters
% resultfile: .bayes.mat file which has the inverse filter params. 
% ix0       : Indices of the area of interest. If it is empty, 
%             all of the dipoles for estimation are included 
%             in the filter. 
%
% --- 
% F       : Inverse filter, I-by-N matrices (I=length(ix)). 
% ix      : Indices of the union of ix0 and indices of the dipoles 
%           caclulated by the original inverse filters.
% meg_filt: Analysis time window. 
% Tperiod : Size of analysis time window
% Tnext   : 
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

load(resultfile,'bayes_parm','Model','basis','Cov',...
     'ix_fine','W','bsnorm');

% NOT support multiple sessions (2005-01-12)
if iscell(basis)
  fprintf(['.bayes.mat file contains estimation results for ' ...
	   '%d sessions\n'],length(basis'));
  fprintf('Only the result of the first session is used.\n');
  basis = basis{1};
  Cov = Cov{1};
  bsnorm = bsnorm(1);
end

if nargin==1 | isempty(ix0)
  ix0 = ix_fine;
end

N = size(Model.a,2); % Number of filters

switch bayes_parm.mode
 case {1,2} % Distributed current estimation
  [ix,ia,ib] = intersect(ix_fine,ix0);
  WW = W(ia,:); % Smoothing filter in interested region
  F = zeros(length(ix),size(Cov,1),N);
  a = size(WW,1)*size(WW,2);
  b = size(F,1)*size(F,2);
  
  for i = 1:size(Model.a,2)
    F0 = meg_filter_old(Model.a(:,i),basis,Cov);
    if b<a
      F(:,:,i) = WW*((1/bsnorm)*F0);
    else
      F(:,:,i) = ((1/bsnorm)*WW)*F0;
    end
  end

 otherwise,
  error(['Estimation mode ' num2str(bayes_parm.mode) ...
	 ' has not been supported.']);
end

% parameters of time windows
meg_filt = bayes_parm.meg_filt;
if isfield(bayes_parm,'Tperiod')
  Tperiod = bayes_parm.Tperiod;
  Tnext = bayes_parm.Tnext; 
elseif length(meg_filt)==2
  Tperiod = meg_filt(2)-meg_filt(1)+1;
  Tnext = Tperiod;
else
  Tperiod = meg_filt(3)-meg_filt(2)+1;
  Tnext = Tperiod;
end



