function [twin_meg, gui_cancelled] = ...
                          job_plot_meg(megfile, mode, twin_lim, twin_ini)
% Plot time course and spatial pattern of MEG data. 
% Return time window which user specified on GUI.
%
% [syntax]
% [twin_meg,gui_cancelled] job_plot_meg(megfile,mode,twin_lim,twin_ini)
%
% [input]
% megfile : <<string>> MEG data file (.meg.mat)
% mode    : <optional> <<int>> 0 for loading average data
%                   1 for loading all trials
%                   2 for intaraction mode(default)
% twin_lim: <optional> <<int vector>> Time window selection limit(meg
%           data index) e.g. twin_lim = [300, 500];
% twin_ini: <optional> <<int vector>> Initial Time window selection(meg
%           data index) (default:[Pretrigger, Pretrigger] from 0msec to
%           0msec) e.g. twin_ini = [300, 500];
%
% [output]
% twin_meg      : <<int vector>> Selected time window(meg data index)
%                 e.g. twin_meg = [300, 500];
% gui_cancelled : <<bool>> gui finish state
%                  = true  : close by pushing x button
%                  = false : close by pushing close button 
%
% [note]
% If nargout==0, "OK" button dissapeared. 
%
% [example]
% >> job_plot_meg('./data/1002425.meg.mat');
% * return selected time window
% >> [twin_meg, gui_cancelled] = job_plot_meg('./data/1002425.meg.mat');
%
% [history]
% 2005-03-19 Taku Yoshioka
% 2006-11-17 Ryosuke Hayashi
% 2007-03-27 Taku Yoshioka
% 2008-09-05 Taku Yoshioka
% 2011-01-19 taku-y
%  [enhancement] Colorscale edit box added.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global vbmeg_inst;
define_job_plot_meg_const;

%
% Input parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~exist('mode', 'var'), mode = 2; end;
if isempty(mode), mode = 2; end;
if ~exist('twin_ini', 'var'), twin_ini = [1 1]; end;
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('twin_lim', 'var'), twin_lim = []; end
  
%
% Create GUI
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
inst_id = create_instance;
if isempty(inst_id), return; end; % Failed to create instance 
h_fig = create_GUI(inst_id, megfile,mode, twin_lim, twin_ini,nargout);
job_plot_meg_update_timecourse(inst_id, FORCE_UPDATE); 
job_plot_meg_update_spatialpattern(inst_id);

% if return value is not specified, return.
if nargout == 0, return; end
%
% wait for application end
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
waitfor(h_fig);

%
% return value
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
twin_meg = vbmeg_inst.plotmeg{inst_id}.twin_meg;
gui_cancelled = vbmeg_inst.plotmeg{inst_id}.gui_cancel;
vbmeg_inst.plotmeg{inst_id} = [];


%%%
%%% Inner function
%%%

function inst_id = create_instance
% Create instance of GUI
% 
% --- Syntax
% function inst_id = create_instance
%
% --- History
% ????-??-?? Taku Yoshioka

global vbmeg_inst

% instance id
if isfield(vbmeg_inst,'plotmeg'),
  for inst_id=1:length(vbmeg_inst.plotmeg)
    % clean up invalid instance
    if isfield(vbmeg_inst.plotmeg{inst_id}, 'h_fig') && ...
       ~ishandle(vbmeg_inst.plotmeg{inst_id}.h_fig)
       vbmeg_inst.plotmeg{inst_id} = [];
    end
    % check container and return empty index as an inst_id
    if isempty(vbmeg_inst.plotmeg{inst_id}), return; end;
  end
  inst_id = length(vbmeg_inst.plotmeg)+1;
else
  % Initialize
  vbmeg_inst.plotmeg = cell(0);
  inst_id = 1; 
end

%%%
%%% Inner function
%%%

function [h_fig] = create_GUI(inst_id,megfile,mode,twin_lim,twin_ini,n)
% Create GUI
% 
% --- Syntax
% function [h_fig] = create_GUI(inst_id,megfile,mode,twin_lim,twin_ini,n)
%
% --- History
% ????-??-?? Taku Yoshioka
% 2008-09-02 Taku Yoshioka

global vbmeg_inst;

%
% Initialize parameters
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% MEG data
plotmeg.megfile = [];
plotmeg.bexp = [];
plotmeg.pick = [];
plotmeg.MEGinfo = [];
plotmeg.T = [];
plotmeg.N = [];
plotmeg.twin_lim = [];
plotmeg.twin_meg = [];
plotmeg.cancel = [];

% Display parameters
plotmeg.n_trial = [];
plotmeg.t = [];
plotmeg.sensor_type = [];
plotmeg.sensor_type_list = [];

% Sensor selection method
plotmeg.sensor_selection_method = 'Brush';

%
% GUI open
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h_fig = open('job_plot_meg.fig');
set(h_fig,'HandleVisibility','on');

% GUI components
plotmeg.h_fig = h_fig;
plotmeg.h_time = get_child_handle(h_fig,'plot_time');
plotmeg.h_space = get_child_handle(h_fig,'plot_space');
plotmeg.ed1 = get_child_handle(h_fig,'trial_number');
plotmeg.ed2 = get_child_handle(h_fig,'time_window_start');
plotmeg.ed3 = get_child_handle(h_fig,'time_window_end');
plotmeg.ed4 = get_child_handle(h_fig,'temporal_peak_size');
plotmeg.ed5 = get_child_handle(h_fig,'edit7');
plotmeg.pb1 = get_child_handle(h_fig,'print_space');
plotmeg.pb2 = get_child_handle(h_fig,'print_time');
plotmeg.sw  = get_child_handle(h_fig, 'save_waveform_push');
plotmeg.pb3 = get_child_handle(h_fig,'move_forward');
plotmeg.pb4 = get_child_handle(h_fig,'move_forward_fast');
plotmeg.pb5 = get_child_handle(h_fig,'move_back');
plotmeg.pb6 = get_child_handle(h_fig,'move_back_fast');
plotmeg.pb7 = get_child_handle(h_fig,'temporal_peak');
plotmeg.cb1 = get_child_handle(h_fig,'fix_colormap');
plotmeg.rb1 = get_child_handle(h_fig,'power_plot');
plotmeg.rb2 = get_child_handle(h_fig,'sensor_plot');
plotmeg.twin_min_range = get_child_handle(h_fig, 'time_window_min');
plotmeg.twin_max_range = get_child_handle(h_fig, 'time_window_max');
plotmeg.close_push = get_child_handle(h_fig, 'close_push');
plotmeg.load_meg_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'load_meg_menu');
plotmeg.loaded_file_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'loaded_file_menu');
plotmeg.meginfo_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'meginfo_menu');
plotmeg.close_menu ...
    = get_child_handle(get_child_handle(h_fig,'file_menu'), ...
                       'close_menu');
plotmeg.sensor_type_menu ...
    = get_child_handle(get_child_handle(h_fig,'view_menu'), ...
                       'sensor_type_menu');


% Set GUI value
set(plotmeg.ed1,'String','1');
plotmeg.gui_cancel = true; % when close button push, turn to false.

%
% Set callback functions
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
inst_str = num2str(inst_id); 
plotmeg_str = ['vbmeg_inst.plotmeg{' inst_str '}'];

% Remove instance after window close
if n == 0
  command = ['global vbmeg_inst; '...
    	   'vbmeg_inst.plotmeg{' inst_str '}=[];'];
  set(h_fig,'DeleteFcn',command);
end

% Show MEG info
command = ['job_plot_meg_show_meginfo(' inst_str ');'];
set(plotmeg.meginfo_menu,'Callback',command);

% Show loaded file
command = ['job_plot_meg_show_loaded_file(' inst_str ');'];
set(plotmeg.loaded_file_menu,'Callback',command);

% Trial number
command = ['job_plot_meg_change_trial(' inst_str ');'];
set(plotmeg.ed1,'Callback',command);

% Time window
command = ['global vbmeg_inst; ' ...
	   plotmeg_str '.t(1)=' ...
	   'str2num(get(' plotmeg_str '.ed2,''String''));' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.ed2,'Callback',command);
command = ['global vbmeg_inst; ' ...
	   plotmeg_str '.t(2)=' ...
	   'str2num(get(' plotmeg_str '.ed3,''String''));' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.ed3,'Callback',command);

% Time window shift
command = ['global vbmeg_inst; '...
	   'dt = ' plotmeg_str '.t(2)-' plotmeg_str '.t(1);' ...
	   plotmeg_str '.t=' plotmeg_str '.t+dt;' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.pb3,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotmeg_str '.t(2)-' plotmeg_str '.t(1);' ...
	   plotmeg_str '.t=' plotmeg_str '.t+5*dt;' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.pb4,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotmeg_str '.t(2)-' plotmeg_str '.t(1);' ...
	   plotmeg_str '.t=' plotmeg_str '.t-dt;' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.pb5,'Callback',command);
command = ['global vbmeg_inst; '...
	   'dt = ' plotmeg_str '.t(2)-' plotmeg_str '.t(1);' ...
	   plotmeg_str '.t=' plotmeg_str '.t-5*dt;' ...
	   'job_plot_meg_update_timecourse(' inst_str ');' ...
	   'job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.pb6,'Callback',command);

% Search peak
command = ['global vbmeg_inst; '...
	   'job_plot_meg_search_peak(' inst_str ');'];
set(plotmeg.pb7,'Callback',command);

% Plot type
command = ['global vbmeg_inst;'...
	   'define_job_plot_meg_const;'...
	   'set(' plotmeg_str '.rb2,''Value'',0);' ...
	   'job_plot_meg_update_timecourse(' ...
	   inst_str ',FORCE_UPDATE);'];
set(plotmeg.rb1,'Callback',command);

command = ['global vbmeg_inst;' ...
	   'define_job_plot_meg_const;' ...
	   'set(' plotmeg_str '.rb1,''Value'',0);' ...
	   'job_plot_meg_update_timecourse(' ...
	   inst_str ',FORCE_UPDATE);'];
set(plotmeg.rb2,'Callback',command);

command = ['job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.cb1,'Callback',command);

% Print figure
command = ['job_plot_meg_print_spatialpattern(' inst_str ');'];
set(plotmeg.pb1,'Callback',command);
command = ['job_plot_meg_print_timecourse(' inst_str ');'];
set(plotmeg.pb2,'Callback',command);

% Save waveform
command = ['job_plot_meg_save_waveform(' inst_str ');'];
set(plotmeg.sw,'Callback',command);

% Close window
command = ['global vbmeg_inst; '...
           'vbmeg_inst.plotmeg{' inst_str '}.gui_cancel = false;',...
           'closereq;'];
set(plotmeg.close_push, 'Callback', command);
if n==0, set(plotmeg.close_push,'Visible','off'); end

% Close window(file->close)
command = ['job_plot_meg_close(' inst_str ')'];
set(plotmeg.close_menu,'Callback',command);

% Close window(x button)
command = ['job_plot_meg_close(' inst_str ', 1)'];
set(plotmeg.h_fig, 'CloseRequestFcn', command);

% Load MEG data
command = ['[meg_dir,meg_file] = vb_file_select' ...
           '({''.meg.mat'',''.eeg.mat''},''Load MEG/EEG data'');' ...
           'if ~isempty(meg_file), ' ...
	   'job_plot_meg_load_meg(' inst_str ',' ...
           '[meg_dir filesep meg_file{1}]); end'];
set(plotmeg.load_meg_menu,'Callback',command);

% Change colorscale
command = ['job_plot_meg_update_spatialpattern(' inst_str ');'];
set(plotmeg.ed5,'Callback',command);

% Set instance
vbmeg_inst.plotmeg{inst_id} = plotmeg;

%
% Load MEG data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(megfile),
  job_plot_meg_load_meg(inst_id, megfile, mode, twin_lim, twin_ini);
end
