function [data] = vb_trial_extractor_tag_by_file(data, tag_spec)
% tag trials by file.
%
% [USAGE]%  [data] = vb_trial_extractor_tag_by_file(data, tag_spec);
% [IN]
%        data : vb_trial_extractor object.
%    tag_spec : tag specification
%         .label_file : text file. This file contains tag list.
%                       each line has one tag.
%                1
%                3
%                2
%                1
%                ...
%         .base_label   : <<cell>> specify tags which is tagged.
%                       example: {'TRIGGER1', 'TRIGGER2'};
%         .overwrite : [true] or false
%                       true : overwrite label without saying.
%                      false : a dialog is shown when overwrite label.
%
% [OUT]
%    data : vb_trial_extractor object.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Previous check
%
if ~exist('data', 'var')
    error('data is a required parameter.');
end
if ~exist('tag_spec', 'var')
    error('tag_spec is not specified.');
end
if ~isfield(tag_spec, 'label_file')
    error('label_file field is not specified.');
end
if exist(tag_spec.label_file, 'file') ~= 2
    error('Specified label file is invalid.');
end
if ~isfield(tag_spec, 'base_label')
    error('base_label field is not specified.');
end
if ischar(tag_spec.base_label)
    tag_spec.base_label = {tag_spec.base_label};
end
if ~isfield(tag_spec, 'overwrite')
    tag_spec.overwrite = true;
end

%
% --- Main Procedure
%

% Read tag file
[label_list] = textread(tag_spec.label_file, '%s %*[^\n]');
new_tags   = unique(label_list);

% Check the number of trials which is specifid tag.
Ntrial = 0;
base_tags = [];
for k=1:length(tag_spec.base_label)
    base_tags = [base_tags; vb_tag_util_find_tag(data.label_list, tag_spec.base_label{k})];
end
base_trial_ix = [];
if ~isempty(base_tags)
    union_base_tag = vb_tag_util_union_tag(base_tags, '', data.trial_list);
    base_trial_ix = vb_tag_get_trial_index(union_base_tag);
    Ntrial = size(base_trial_ix, 1);
end

Ntag      = length(label_list);
if Ntrial ~= Ntag
    errordlg(...
     sprintf('The number of trials(%d) and labels(%d) written in label file are different. Check label file.', ...
             Ntrial, Ntag), 'error');
    return;
end

% Duplicate check between new tags and exist tags
exist_tags = cell(0);
for k=1:length(data.label_list)
    exist_tags{k} = vb_tag_get_tag_name(data.label_list{k});
end

dup_tags = cell(0);
dup_ix   = [];
for k=1:length(new_tags)
    ix = strmatch(new_tags{k}, exist_tags, 'exact');
    if ~isempty(ix)
        dup_tags{k} = new_tags{k};
        dup_ix      = [dup_ix; ix];
    end
end

if length(dup_tags) && tag_spec.overwrite == false
   msg = sprintf('Following tags already exists, Continue?\n');
   for k=1:length(dup_tags)
       msg = [msg, sprintf('%s\n', dup_tags{k})];
   end
   res = questdlg(msg, 'Confirm', 'Yes', 'No', 'Yes');
   if strcmp(lower(res), 'no')
       return;
   else
       data.label_list(dup_ix) = []; % remove duplicate tag.
   end
end

% Create new tag and register trial index
for k=1:length(new_tags)
    tag_obj = vb_tag_new(new_tags{k});
    

    % extract data index
    list_ix = strmatch(new_tags{k}, label_list, 'exact');
    data_ix = base_trial_ix(list_ix);

    % register trial
    tag_obj = vb_tag_add_trial_index(tag_obj, data_ix);
    data.label_list{length(data.label_list)+1} = tag_obj;

    vb_disp(sprintf('created label:%s, trial=%d', new_tags{k}, length(data_ix)));
end

data.label_spec = tag_spec;

%
% --- After check
%
if nargout ~= 1
    error('function caller should receive vb_trial_extractor object.');
end
