function Tsurf = vb_voximg2surimg(XYZvox,Tvox,XYZsurf, r2thres, mode)
% MAP VOXEL IMAGES TO SURFACE IMAGE USING INVERSE DISTANCE WEIGHTING
% INTERPOLATION OR NEAREST NEIGHBOR SELECTION
%
% [Input]
% XYZvox : voxel image coordinates with unit mm [Nvox,3]
% Tvox   : voxel value [Nvox,1]
% XYZsurf : coordinates of surface mesh points with unit mm [Nv,3] 
% r2thres : distance threshold with unit mm^2 
% 
% [Optional]
% mode (=1)  : flag to switch mapping mode
%              1 = inverse distance weighting interpolation
%              2 = nearest neighbor selection
%
% [Output]
% Tsurf  : mapped surface image
%
%
% 2016/07/06 O.Yamashita
% * add eps to distance calculation to prevent inverse of distance from
% being infinity
% 2016/07/01 O.Yamashita
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if nargin < 5
    mode = 1;
end

Nv = size(XYZsurf,1);
Tsurf = zeros(Nv,1);

switch mode
    case 1,
        fprintf('  inverse distance weighting interpolation within sphere of radius %2.2f mm ...\n', sqrt(r2thres));

        for vv = 1: size(XYZsurf,1),
            XYZref = XYZsurf(vv,:);

            % square distance and inverse square distance
            r2 = (XYZvox(:,1)-XYZref(1)).^2+ (XYZvox(:,2)-XYZref(2)).^2+ (XYZvox(:,3)-XYZref(3)).^2 +eps;
            ir2 = 1./r2;

            % inverse distance weighting
            ixvox = find(r2 < r2thres);
            if ~isempty(ixvox)  % voxels found within sphere of radius r2thres
                Tsurf(vv) = Tvox(ixvox)' * ir2(ixvox)/sum(ir2(ixvox));
            end

            if mod(vv,1000)==0, fprintf('%d..', vv); end

        end
        
    case 2
         fprintf('  nearest neighbor selection within sphere of radius %2.2f mm ...\n', sqrt(r2thres));
         
         for vv = 1: size(XYZsurf,1),
             XYZref = XYZsurf(vv,:);
             
             % square distance and inverse square distance
             r2 = (XYZvox(:,1)-XYZref(1)).^2+ (XYZvox(:,2)-XYZref(2)).^2+ (XYZvox(:,3)-XYZref(3)).^2;
            
             % nearest neighbor
             [r2min,jx] = min(r2);
             if r2min < r2thres
                 Tsurf(vv) = Tvox(jx);
             end
             if mod(vv,1000)==0, fprintf('%d..', vv); end
         end
   
    otherwise 
        error('mode must be 1 or 2 !');
            
end