function [Vout, err, iter] = ...
	vb_dipole_search(Vinit, bexp, pick, Qpick, Wsensor, errfunc, Option, mode)
% Dipole position serach by minimizing error
%   [Vout, err, iter] = ...
%		vb_dipole_search(Vinit, bexp, pick, Qpick, Wsensor, errfunc, Option, mode)
% --- Output
% Vout : Estimated position
% err  : residual normalized error
% iter : iteration number
%
% --- INPUT
% errfunc : error function to calculate dipole estimation error
% Vinit      : initial dipole position (Ndipole x 3)
% Vinit(n,:) : dipole position (3-D coordinate) at n-th vertex
%
% bexp   : MEG-data   (Nsensor ,Time) 
%
%  pick(k, 1:3) : Sensor coil position  : , 
% Qpick(k, 1:3)	: Sensor coil direction : 
%
% Wsensor(m,n) = n-th coil weight for m-th sensor channel
%    basis(channel,dipole) = Wsensor * basis(coil,dipole)
%
% Option  : Option in optimization
% mode  : Minimization mode 
%    =0 : fminsearch
%    =1 : fminunc
%
% 2006-12-16 made by M.Sato
% 2007-4-3 made by M.Sato
% ---
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

Vinit = Vinit(:);

% error function name for dipole search
if ~exist('errfunc','var')
	errfunc = 'vb_dipole_error';
end
%
% Set minimization option
%
% Optimization setting
if isfield(Option,'MaxIter') & ~isempty(Option.MaxIter)
	MaxIter = Option.MaxIter; 
else
	MaxIter = 5000; 
end;

if isfield(Option,'TolFun') & ~isempty(Option.TolFun)
	TolFun = Option.TolFun; 
else
	TolFun = 1.e-10; 
end;

if isfield(Option,'TolX') & ~isempty(Option.TolX)
	TolX = Option.TolX; 
else
	TolX = 1.e-8; 
end;

if isfield(Option,'Display') & ~isempty(Option.Display)
	Display = Option.Display; 
else
	Display = 'off';
end;

OPTIONS = optimset('MaxIter', MaxIter, ...
                   'MaxFunEvals', 2*MaxIter,...
				   'TolX',    TolX, ...
				   'TolFun',  TolFun, ...
                   'Display', Display, ...
                   'LargeScale', 'off', ...
                   'GradObj','off'); 

% Minimize the error function

BB = bexp*bexp';	% BB(Nsensor,Nsensor)

% Search method
if ~exist('mode','var') | isempty(mode)
	mode = vb_tool_check('Optimization');
	%	 = 0;	% Use 'fminsearch'
	%	 = 1;	% Use 'fminunc' in Optimization Toolbox
end

switch	mode
case	0
	% Nelder-Mead Simplex Method 
	disp('Nelder-Mead Simplex Method');
	[Vout, err, flag, outputs]= ...
	    fminsearch(errfunc,Vinit, OPTIONS, BB, pick, Qpick, Wsensor); 
case	1
	% Unconstrained optimization 
	% Optimization toolbox
	disp('Unconstrained optimization')
	[Vout, err, flag, outputs]= ...
		fminunc(errfunc,Vinit, OPTIONS, BB, pick, Qpick, Wsensor); 
case	2
	% Unconstrained optimization 
	disp('Unconstrained optimization')
	[Vout, err, flag, outputs]= ...
		fminunc1(errfunc,Vinit, OPTIONS, BB, pick, Qpick, Wsensor); 
end

iter = outputs.iterations;

NV	  = length(Vout(:))/3;		% Number of dipoles
Vout  = reshape(Vout,[NV 3]);
