function [a0,Ta0] = vb_set_vb_prior(prior_parm)
% Return prior hyper-parameter using fMRI activity pattern. 
% 
% [syntax]
% [a0,Ta0] = vb_set_vb_prior(prior_parm)
%
% [input]
% prior_parm: <<struct>> Prior specification parameters. 
% ---fields of prior_parm
%  .ix     : <<int vector>>
%  .a0     : <<float vector>> Normally equal to 1 [1]. 
%  .a0_act : <<float vector>> Magnification parameter [1].
%  .Ta0    : <<float vector>> gamma0 at minimum fMRI signal [2]. 
%  .Ta0_act: <<float vector>> gamma0 at maximum fMRI signal [2]. 
%  .v0     : <<float>> Noise-normalized current variance estimated by
%            minimum norm estimation. 
% ---
% Model     : <<struct>> <optional> Model parameter set. 
%
% [output]
% a0 : <<float vector>> Noise-normalized current variance, alpha0^-1
%      [2]. 
% Ta0: <<float vector>> Confidence parameter, gamma0 [2]. 
%
% [references]
% [1] Yoshioka et al., 2008. 
% [2] Sato et al., 2004. 
%
% [history]
% 2008-06-27 Taku Yoshioka
% 2010-09-16 Taku Yoshioka
%  Bug fix (line 41): It doesn't have any effect if previously obtained
%  'Model' is not used to set prior parameters. 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% If Model is given, Ta0 is calculated using 'Model.a'
if isfield(prior_parm,'Model') & ~isempty(prior_parm.Model), 
  disp('Set prior by Wiener estimation result')
	
  a0 = Model.a;
  a0 = sum(a0,2)/size(a0,2);
  Ta0 = Ta0 + (Ta0_act-Ta0)*(a0./max(a0));
else % If Model is not given, a0 & Ta0 is calculated using fMRI
  disp('Set prior fMRI activity pattern'); 

  % prior parameters
  v0       = prior_parm.v0;
  a0       = prior_parm.a0*v0; 
  a0_act   = prior_parm.a0_act*v0; 
  Ta0      = mean(prior_parm.Ta0); 
  Ta0_act  = mean(prior_parm.Ta0_act); 
  act      = prior_parm.act;
  
  % fMRI information
  xxP = act.xxP/max(act.xxP);
  xxP = xxP(prior_parm.ix);
	
  a0 = a0 + (a0_act-a0)*(xxP.^2); % square
  Ta0 = Ta0 + (Ta0_act-Ta0)*(xxP);  % linear
  %Ta0tmp = Ta0_act*ones(Nvact,1); 	% Constant
end;

return;
