function [B, Model, nGact, nGall, nGGall, Cov, bsnorm, ix_act, ix_bck, ...
             vb_parm, Wact, Wbck, ix_act_ex, ix_bck_ex] ...
         = vb_load_result_z(proj_root, resultfile, ix_area)
% Load VB estimation result and necessary data for current reconstruction
% [B, Model, nGact, nGall, nGGall, Cov, bsnorm, ix_act, ix_bck, ...
%     vb_parm, Wact, Wbck, ix_act_ex, ix_bck_ex] ...
%         = vb_load_result(proj_root, resultfile, ix_area)
% --- Input
% proj_root
% resultfile : result file obtained by VB estimation
% ix_area : Position index to calculate estimated current
%    If 'ix_area' is empty or not given, 
%    currents in the focal region are calculated
%
% 2006-08-31 M. Sato
% 2006-11-11 M. Sato
% If 'ix_area' is empty, original focal area is used
% instead of extended focal area by spatial smoothing
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


if ~isempty(proj_root)
  resultfile = fullfile(proj_root, resultfile);
end  

load(resultfile, 'bayes_parm','Model','Cov','vb_parm');

if ~isempty(proj_root);
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

if isfield(bayes_parm_abs,'remove_crossed_area'),
	remove_crossed_area = bayes_parm_abs.remove_crossed_area;
else
	remove_crossed_area = OFF;
end

% MEG data preparation
[B] = vb_megdata_preparation(bayes_parm_abs);


% Preparation of lead fields
[nGall, bsnorm, ix_global, ix_global_ex, Wbck, Lbck] = ...
	vb_global_leadfield_preparation(bayes_parm_abs);
[nGact, ix_focal, ix_focal_ex, Wact, Lact]  =  ...
	vb_focal_leadfield_preparation(bayes_parm_abs, bsnorm);

if remove_crossed_area == ON,
	% Remove focal area from global area
	switch bayes_parm_abs.forward_model
	case 'focal+global',
		[nGall, bstmp, ix_global, ix_global_ex, Wbck, Lbck] = ...
			vb_global_leadfield_preparation(bayes_parm_abs, bsnorm, ON );
	case 'focal',
		ix_global = [];
		ix_global_ex = [];
	end
end

%%%% Find active current area

% Current variance
A_inv = sum(Model.a , 2);	% sum over all time window
Nvact = vb_parm.Nvact;	    % # of active vertex 
Nact  = length(A_inv);
L     = Nact/Nvact;
Lact  = vb_parm.Norient;

if bayes_parm_abs.variance_orientation == ON
	A_inv = reshape(A_inv, [Nvact  L]);
	A_inv = sum( A_inv ,2);
end

% Find nonzero component of (A_inv) : active current region
ix_act_nz   = find( A_inv > 0 );
ix_focal_nz = ix_focal(ix_act_nz);

% Area in which current is calculated
if isempty(ix_area),
	ix_area = ix_focal;		% original focal area
%	ix_area = ix_focal_nz;	% extended focal area by spatial smoothing
end

% Select vertex index 'ix_area' within the active current region
[j_tmp, ix_area_nz] = vb_index2indexsub(ix_area, ix_focal_nz);

%%%% Area information where the currents are reconstructed.

% Convert vertex index to index within the active current
% Indices outside 'ix_focal' are removed from current indices 'ix_area'

% ix_act : current indices w.r.t all vertices
% j_act  : current indices 'ix_act' w.r.t ix_focal
[j_act, ix_act] = vb_index2indexsub(ix_area_nz, ix_focal);

% Global(background) area is independent of active current region
if remove_crossed_area == OFF
	% select active region
	[k_bck, ix_bck] = vb_index2indexsub(ix_area_nz, ix_global);
else
	[k_bck, ix_bck] = vb_index2indexsub(ix_area, ix_global);
end

%Njact	 = vb_parm.Njact;	        % # of active current parameters

% # of focal window
Nvact_area = length(ix_act);	    % # of active vertex in area
Njact_area = Nvact_area * Lact;     % # of current parameters in area

% # of global window 
Lbck  = vb_parm.Norient_all;
Nvall = length(ix_global);         % # of dipoles in the background 
Nvall_area = length(ix_bck);       % # of background points in area
Njall_area = Nvall_area* Lbck ;    % # of current parameters in area

% Extract active current from a_inv
a_inv = Model.a;
Nwin  = size(a_inv,2);

if bayes_parm_abs.variance_orientation == ON
	jj_act = repmat(j_act(:), [1 L]) ...
	       + repmat((0:(L-1))*Nvact, [Nvact_area 1]);
else
	jj_act = j_act;
end

Model.a = a_inv(jj_act(:),:);

vb_parm.Njact_area = Njact_area;
vb_parm.Njall_area = Njall_area;

Nsession = vb_parm.Nsession;	% Number of sessions

% Extract active current from nGact
jj_act = repmat(j_act(:), [1 Lact]) ...
       + repmat((0:(Lact-1))*Nvact, [Nvact_area 1]);
kk_bck = repmat(k_bck(:), [1 Lbck]) ...
       + repmat((0:(Lbck-1))*Nvall, [Nvall_area 1]);

for n = 1 : Nsession
	G  = nGact{n};
	Gb = nGall{n};

	nGact{n}  =  G(:, jj_act(:));
	nGall{n}  = Gb(:, kk_bck(:));
 	nGGall{n} = Gb * Gb';    
end

%% Restrict smoothing filter 'W' to estimation points
Wact = Wact(:,j_act);	
Wbck = Wbck(:,k_bck);   

j_act_ex = find( sum(Wact,2) > 0);
k_bck_ex = find( sum(Wbck,2) > 0);

% ix_act_ex : current indices w.r.t all vertices
% j_act_ex  : current indices w.r.t ix_focal_ex

ix_act_ex = ix_focal_ex(j_act_ex);
ix_bck_ex = ix_global_ex(k_bck_ex);

Wact = Wact(j_act_ex,:);
Wbck = Wbck(k_bck_ex,:);

fprintf('# of active vertex   = %d\n', Nvact_area)
fprintf('# of active current  = %d\n', Njact_area)

fprintf('# of background vertex  = %d\n', Nvall_area)
fprintf('# of background current = %d\n', Njall_area)

fprintf('# of extended active vertex   = %d\n', length(ix_act_ex))
fprintf('# of extended background vertex   = %d\n', length(ix_bck_ex))

% ix_act : Vertex index corresponding to active current Zact
% ix_bck : Vertex index corresponding to background current Zbck
% ix_act_ex : Vertex index corresponding to active current Jact
% ix_bck_ex : Vertex index corresponding to background current Jbck
% Wact   : Spatial smoothing matrix of focal window
% Wbck   : Spatial smoothing matrix of global window
% Jact   = Wact * Zact
% Jbck   = Wbck * Zbck

% [Njact, Tall, Ntry] = size(Zact);
% Njact_ex = size(Wact,1);
% 
% Jact  = Wact * reshape(Zact,  [Njact/Lact, Lact*Tall*Ntry]); 
% Jact  = reshape(Jact,  [Njact_ex*Lact, Tall, Ntry]); 
