function [Zact,Jinfo,bayes_parm,vb_parm,MEGinfo,Jext,Pointlist] ...
    = vb_current_reconstruct_z_dynamics(proj_root, curr_parm)
% Current reconstruction using Bayesian inverse filter.
%
% [syntax]
%  [Zact,Jinfo,bayes_parm,vb_parm,MEGinfo,Jext,Pointlist] ...
%          = vbmeg_current_reconstruct_z2(proj_root, curr_parm)
%
% [input]
% proj_root: <<string>> VBMEG project root directory. 
% curr_parm: <<struct>> Parameters for current estimation.
% --- fields of curr_parm
%  dbayesfile   : <<string>> Dynamics estimation file (.dbayes.mat). 
%  currfile     : <<string>> Cortical current file (.curr.mat), created
%                 by this function. 
%  trial_average: <optional> <<bool>> If true, 
%                 = [ON] : average current over all sessions
%                 = OFF  : current for each session
%  ix_area      : <optional> Vertex indices to calculate estimated 
%                 current. If 'ix_area' is empty or not given, cortical
%                 currents in the active region are calculated. 
%  tsubsmpl     : <optional> <bosolete> Specify subsampled time
%                 index. If 'tsubsmpl' is empty or not given, time
%                 subsampling is not done. 
%  dsampf       : <optional> <<int>> Specify frequency of
%                   downsampling. This value must be smaller than the
%                   original sampling frequency of M/EEG data. 
%  overlap_mode : <optional> <<int>>
%   = 0 : current is averaged over overlapped time window
%   = 1 : current is not averaged for overlapped window
%         current time series of each time windows 
%         are concatenated sequentially for spectral analysis
%  verbose      : <<bool>> Verbose flag. 
% ---
%
% [note] !! currently unconfirmed.
%        If following field is given, these values are used instead of
%        bayes_parm field in result file:
% ---
%  curr_parm.basisfile
%  curr_parm.megfile  
%  curr_parm.twin_meg 
%  curr_parm.Tperiod  
%  curr_parm.Tnext
%  curr_parm.extra.basisfile (for extra dipole)
% ---
%
% [output]
% Zact    : active current
%
% Zact(n,t,:) is the current at the vertex 'ix_act(n)' & the time 't'
% Zact(Nact,Nsample)          for trial_average = ON 
% Zact(Nact,Nsample,Ntrials)  for trial_average = OFF
%   Nact     : # of active region, 
%   Nsample  : # of time sample, 
%   Ntrials  : # of trials in all session]
% --- fields of Jinfo
%  version   : <<string>> Version of cortical current file.
%  curr_type : <<string>> 'Z-current'. It can be 'J-current' for VBMEG
%              version 0.8 or older. 
%  Wact      : <<float matrix>> Smoothing Gaussian filter, mapping from
%              Z-current to J-current. 
%  ix_act    : <<int vector>>: Vertex indices of Z-current.
%  ix_act_ex : <<int vector>>: Vertex indices of J-current.
%  Lact      : <<int>> Number of current direction at one vertex. 
%  Tsample   : <<int vector>> Time sample indices of the original MEG
%              data. length(Tsample) == size(Zact,2) == size(Jact,2). 
%  Tmsec     : <<float vector>> Time in msec. 
%  SampleFreq: <<float>> Sample frequency of cortical current, not
%              original M/EEG signal [Hz]. 
%  Pretrigger: <<int>> Time points of the length of the pretrigger
%              period of cortical current data. It is neither actual time
%              nor time points of the original M/EEG signal. 
%  Ntrial    : <<int>> Number of trials of estimated current. 
%  patch_norm: <<bool>> Cortical current is patch size normalized
%              (Am/m^2) or not (Am). 
%  Tix       : <<L x 1 cell>> Time sample indices of each time window. 
%              Zact(:,Tix{n},:) is the set of Z-current within the n-th
%              time window.
% ---
%
% * Inverse filter calculation is done in vb_invfilter_calc
%
% [history]
% 2006-09-03 M. Sato
% * Non-overlapped concatenation mode is added for spectral analysis
% 2008-08-19 Taku Yoshioka
%   Extra dipole support
% 2008-09-30 Taku Yoshioka
%   Minor change for variables in Jinfo
% 2008-10-23 Taku Yoshioka
%  Bug fix for current estimation without extra dipoles
% 2009-04-02 Taku Yoshioka
%  Parameter name changed within this code for readability
%  (just replacing 'resultfile' to bayesfile)
% 2010-03-01 M. Sato
%  Bug fix for Wact index and fieldname(tsubsamp->tsubsmpl)
% 2010-12-06 taku-y
%  [enhancement] curr_parm.dsampf supported. 
%  [minor]       Following fields of Jinfo set in this function: 
%                 SampleFreq
%                 Pretrigger
%                 Tmsec
%  [trivial]     Jinfo.version = vb_version. 
% 2010-12-07 taku-y
%  [trivial] Jinfo.version = vbmeg('version');
% 2011-05-11 takiu-y
%  [debug] Jinfo.Tmsec corrected.
% 2011-06-28 taku-y
%  [minor] Jinfo.Tix added.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

if ~isempty(proj_root)
  dbayesfile = fullfile(proj_root, curr_parm.dbayesfile);
else
  dbayesfile = curr_parm.bayesfile;
end


%
% Verbose level setting
% (note: 'verbose_level' is not related to input variable
% 'curr_parm.verbose'. So two configurations relating to message display
% are coexisting in this function.)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
global vbmeg_inst
verbose_const = vb_define_verbose; 

if isempty(vbmeg_inst) | ~isfield(vbmeg_inst,'verbose_level'), 
  verbose_level = verbose_const.VERBOSE_LEVEL_NOTICE;
else
  verbose_level = vbmeg_inst.verbose_level;
end

%
% load VBMEG estimated result
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
load(dbayesfile, 'Model','vb_parm','dbayes_parm');
bayes_file = fullfile(proj_root, dbayes_parm.bayes_file);
load(bayes_file, 'bayes_parm', 'Model_ext', 'Pointlist');

%
% check parameter of 'curr_parm'
%
% Values of 'curr_parm' fields dominates over 
%   those of 'bayes_parm' in bayesfile
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[bayes_parm,ix_area,trial_average,tsubsamp,overlap_mode,dsampf] ...
    = check_arg(bayes_parm, curr_parm);

if ~isempty(proj_root);
  bayes_parm_abs = vb_parm_absolute_path(proj_root, bayes_parm);
else
  bayes_parm_abs = bayes_parm;
end

% create temp area file and add diffusion parcellation area
% to the copied area file. then use it instead of original area file.
area_key = 'dmri_parcellation_area';
[bayes_parm_abs, dbayes_parm] = ...
    inner_replace_area_file_with_diffusion_area_file(...
                                    bayes_parm_abs, dbayes_parm, area_key);

%
% MEG data preparation
% B      : MEG data
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[B,Ntrials,Nch,Tsample,Twindow,Tmsec] ...
    = vb_megdata_preparation(bayes_parm_abs);
MEGinfo = vb_load_meg_info(bayes_parm_abs.megfile{1});

%
% Preparation of lead fields
% Gact   : leadfield of focal window
% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Focal window
vb_disp('--- Lead field matrix of focal window');

lf_parm.brainfile = bayes_parm_abs.brainfile;
lf_parm.areafile = bayes_parm_abs.areafile;
lf_parm.patch_norm = bayes_parm_abs.patch_norm;
lf_parm.expand_spatial_filter = bayes_parm_abs.expand_spatial_filter;
lf_parm.spatial_smoother = bayes_parm_abs.spatial_smoother;
lf_parm.basisfile = bayes_parm_abs.basisfile;
lf_parm.area_key = bayes_parm_abs.area_key;
lf_parm.reduce = bayes_parm_abs.reduce;
lf_parm.Rfilt = bayes_parm_abs.Rfilt;
lf_parm.remove_area_key = [];

[Gact, ix_act, ix_act_ex, Wact, Lact] = ...
    vb_leadfield_preparation(lf_parm);

% Extra dipole
if isfield(bayes_parm_abs,'extra') & ~isempty(bayes_parm_abs.extra), 
  vb_struct2vars(bayes_parm_abs,{'extra'});
  fprintf('--- Lead field matrix of extra dipoles \n');
  for n=1:length(extra.basisfile)
    tmp = vb_load_basis(extra.basisfile{n});
    Gext{n} = tmp';
  end
else
  Gext = [];
end

%
% Area index in which current is calculated
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isempty(ix_area),
  % Select vertex index 'ix_area' within the active current region
  [jx_area_ex, ix_area_ex] = vb_index2indexsub(ix_area, ix_act_ex);
else
  jx_area_ex = 1:length(ix_act_ex);
end

Wact   = Wact(jx_area_ex,:);
jx_act = find( sum(Wact, 1) > 0);
Wact   = Wact(:,jx_act);

% active index of Z-current
ix_act = ix_act(jx_act);
% active index of J-current
ix_act_ex = ix_act_ex(jx_area_ex);

% # of active vertex
Njact_area = length(jx_act);

% # of extra dipoles
if ~isempty(Gext), Njext = size(Gext{1},2);
else Njext = 0; end

%
% Constant
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Nsession = length(B);	 % Number of sessions
Ntotal   = sum(Ntrials); % Total number of trials in all sessions

%
% Temporal subsampling index
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
sampf = MEGinfo.SampleFreq;

if ~isempty(dsampf), 
  tsubsamp = ceil(1:sampf/dsampf:Tsample);
  Jinfo.SampleFreq = dsampf;
else
  if isempty(tsubsamp), tsubsamp = 1:Tsample; end
  Jinfo.SampleFreq = sampf;
end

%Jinfo.Tmsec = Tmsec(tsubsamp);
[tmp,ix] = min(abs(Tmsec(tsubsamp)));
Jinfo.Pretrigger = ix;

%
% Temporal smoothing window weight
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[Tweight ,Tindex, Nindex, Nsample] = ...
    vb_overlapped_timewindow_weight(Twindow, Tsample, tsubsamp, overlap_mode);

Nwindow   = length(Nindex);   	% # of time window
Jinfo.Tix = Nindex;

if overlap_mode == 1,
  vb_disp('Non-overlapped concatenation mode'); 
end

%
% Initialization
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if trial_average == ON
  % Current averaged over trials
  Zact = zeros(Njact_area,Nsample);
  Jext = zeros(Njext,Nsample);
else 
  % Current for each trial
  Zact = zeros(Njact_area,Nsample,Ntotal);
  Jext = zeros(Njext,Nsample,Ntotal);
end

%
% Estimated current variance
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
a_inv = Model.a;
Cov = Model.Cov;
if ~isempty(Model_ext), e_inv = Model_ext.a;
else e_inv = []; end

%
% Store current sources in bayes.mat
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if trial_average == ON
  if Nsession == 1
    if size(Model.Z{1},3) == 1
      Zact = Model.Z{1};
      Jext = [];
    else
      error('The number of trials must be one.');
    end
  else
    error('The number of sessions must be one.');
  end
else
  error('Trials must be averaged.');
end

%
% Current Info
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Jinfo.version   = '2.0-0.b.9'; %vbmeg('version');
Jinfo.curr_type = 'Z-current';

Jinfo.Lact      = Lact;
Jinfo.Wact      = Wact;
Jinfo.ix_act    = ix_act;
Jinfo.ix_act_ex = ix_act_ex;
Jinfo.NJact     = Njact_area;
Jinfo.Nsession  = Nsession;
Jinfo.jactdir   = [];
Jinfo.Ntrial    = Ntrials; 

% ix_act : Vertex index corresponding to active current Zact
% ix_act_ex : Vertex index corresponding to active current Jact
% Wact   : Spatial smoothing matrix of focal window
% Jact   = Wact * Zact

% Actual time corresponding to columns of Zact, supporting overlap mode
% and non-overlapped concatenation mode
Tid_all = [];
Nid_all = [];
for j=1:Nwindow
  Tid_all = [Tid_all Tindex{j}];
  Nid_all = [Nid_all Nindex{j}];
end

if overlap_mode==false,
  ix          = unique(Tid_all);
  Jinfo.Tmsec = Tmsec(ix);
else
  Jinfo.Tmsec = Tmsec(Tid_all);
end

%Tstart  = bayes_parm.twin_meg(1);
%Tend    = bayes_parm.twin_meg(2);
%if isempty(tsubsamp)
%  Jinfo.Tsample = Tstart:Tend;
%else
%  Jinfo.Tsample = tsubsamp + Tstart - 1;
%end

return;

%%%% ---------------
function [bayes_parm, ix_area, trial_average, tsubsamp, overlap_mode,dsampf] = ...
    check_arg(bayes_parm,curr_parm)

if isfield(curr_parm,'basisfile'), 
  bayes_parm.basisfile = curr_parm.basisfile;
end;
if isfield(curr_parm,'megfile'), 
  bayes_parm.megfile   = curr_parm.megfile  ;
end;
if isfield(curr_parm,'twin_meg'), 
  bayes_parm.twin_meg  = curr_parm.twin_meg ;
end;
if isfield(curr_parm,'Tperiod'), 
  bayes_parm.Tperiod   = curr_parm.Tperiod  ;
end;
if isfield(curr_parm,'Tnext'), 
  bayes_parm.Tnext     = curr_parm.Tnext    ;
end;

if ~isfield(curr_parm,'trial_average'), 
  trial_average = ON; 
else
  trial_average = curr_parm.trial_average; 
end;

bayes_parm.trial_average = trial_average;

if ~isfield(curr_parm,'ix_area'),  
  ix_area = []; 
else
  ix_area = curr_parm.ix_area; 
end;
if ~isfield(curr_parm,'tsubsmpl'), 
  tsubsamp = []; 
else
  tsubsamp = curr_parm.tsubsmpl; 
end;
if ~isfield(curr_parm,'overlap_mode'),     
  overlap_mode = 0; 
else
  overlap_mode = curr_parm.overlap_mode; 
end;

if isfield(curr_parm,'extra'),
  if isfield(curr_parm.extra,'basisfile'), 
    bayes_parm.extra.basisfile = curr_parm.extra.basisfile;
  end;
end;

if ~isfield(curr_parm,'dsampf'), 
  dsampf = [];
else
  dsampf = curr_parm.dsampf;
end;

return;

function [bayes_parm_abs, dbayes_parm] = ...
    inner_replace_area_file_with_diffusion_area_file(...
                                                bayes_parm_abs, ...
                                                dbayes_parm, area_key)
% copy specified area file as a temporary file and dMRI are into the file,
% then return the filename.
% [Input]
%    bayes_parm_abs  : initial value is calculated by this bayes_parm.
%    dnayes_parm     : dynamics bayes estimation parameter.
%    area_key        : save dmri area into a temporary file by this name.
% [Output]
%    bayes_parm_abs.areafile : temporary area filename including dMRI area.

if ~isfield(dbayes_parm, 'area_key') || isempty(dbayes_parm.area_key)
    dbayes_parm.area_key = area_key;
end
if ~isfield(dbayes_parm, 'area_key_global') || isempty(dbayes_parm.area_key_global)
    dbayes_parm.area_key_global = area_key;
end

% find area key
found = false;
load(bayes_parm_abs.areafile, 'Area');
for k=1:length(Area)
    if strcmp(Area{k}.key, area_key)
        found = true;
    end
end
if found, return; end

% bayes_parm_abs.areafile 
load(dbayes_parm.dmri_file, 'vbmeg_area_ix');

% copy area file to temp area file and put dMRI area into the file.
temp_area_file = [tempname(tempdir), '.area.mat'];
copyfile(bayes_parm_abs.areafile, temp_area_file);
area_new.Iextract = vbmeg_area_ix;
area_new.key      = area_key;
vb_add_area(temp_area_file, area_new);

% use temp_area_file instead of original area file.
bayes_parm_abs.areafile = temp_area_file;
