function [trigger_channel_idx, trigger_sample_idx] = ...
  vb_msrmnt_get_trigger_samples(msrmnt_file, trigger_ch, tr_strt_idx)
% return indices of trigger sample from measurement file and specified 
% trigger channels
% [usage]
%   [trigger_channel_idx, trigger_sample_idx] = ...
%     vb_msrmnt_get_trigger_samples(msrmnt_file, trigger_ch, tr_strt_idx)
% [input]
%   msrmnt_file : <required> measurement file
%               : EEG-MAT(.eeg.mat) or MEG-MAT(.meg.mat = unsupported yet)
%    trigger_ch : <required> cell array of trigger channel labels
%   tr_strt_idx : <optional> trigger starting index [auto-detect]
% [output]
%   trigger_channel_idx : array of trigger channel index
%    trigger_sample_idx : <struct> information box of trigger [1 x N_channel]
%               : all_idx : index of all samples
%               : beg_idx : index of beginning samples
%               : end_idx : index of ending samples
% [note]
%   
% [history]
%   2007-01-04 (Sako) initial version
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%   2011-07-29 (Sako) replaced searching index function
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

vb_define_device;

% --- CHECK ARGUMENTS --- %
if ~exist('msrmnt_file', 'var'), msrmnt_file = []; end
if ~exist('trigger_ch', 'var'),  trigger_ch = []; end
if ~exist('tr_strt_idx', 'var'), tr_strt_idx = []; end
[msrmnt_file, trigger_ch, tr_strt_idx, ...
    Measurement, info, data, trg_ch_idx] = ...
  inner_check_arguments(msrmnt_file, trigger_ch, tr_strt_idx);

trigger_channel_idx = trg_ch_idx;

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;
switch Measurement
  case  'EEG'
    idx = vb_eeginfo_get_trigger_status_label(info);

    status_data = data(idx,:);

    bit_len = DEFAULT_BIT_LEN;
    bit_arrays = vb_eeg_make_channel_bit_array(status_data, trg_ch_idx, bit_len);

    trigger_sample_idx = [];
    for nch = 1:size(bit_arrays,1)
      trigger_sample_idx = [trigger_sample_idx; ...
        vb_util_get_bit_on_list(bit_arrays(nch,:), bit_len)];
    end

  case  'MEG'
    error('(%s)not be implemented yet for ''%s''', func_, Measurement);
  otherwise
    error('(%s)unknown Measurement: ''%s''', func_, Measurement);
end
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [msrmnt_file, trigger_ch, tr_strt_idx, ...
    Measurement, info, data, trg_ch_idx] = ...
  inner_check_arguments(msrmnt_file, trigger_ch, tr_strt_idx)

func_ = mfilename;

info = [];
data = [];
trg_ch_idx = [];

if isempty(msrmnt_file)
  error('(%s)msrmnt_file is a required parameter', func_);
end

if exist(msrmnt_file, 'file') ~= 2
  error('(%s)cannot read msrmnt_file : %s', func_, msrmnt_file);
end

[measurement] = vb_load_device(msrmnt_file);
Measurement = upper(measurement);

switch Measurement
  case  'EEG'
    info = vb_eegfile_load_eeginfo(msrmnt_file);
    data = load_eeg_data(msrmnt_file);

    if isempty(tr_strt_idx)
      tr_strt_idx = vb_eeginfo_get_sensor_number(info) + 1;
    end;

    % index of trigger channel
    all_ch_label = vb_eegfile_get_channel_label(msrmnt_file);
    org_ch_idx = vb_util_get_matching_label_index(all_ch_label, trigger_ch);
    trg_ch_idx = org_ch_idx - tr_strt_idx + 1;
    
    % re-arrange [1 x N] is expected in this function
    trg_ch_idx = vb_util_arrange_list(trg_ch_idx,1);
    
  case  'MEG'
    error('(%s)not be implemented yet for ''%s''', func_, Measurement);
  otherwise
    error('(%s)unknown Measurement : ''%s''', func_, Measurement);
end

if isempty(trigger_ch)
  error('(%s)trigger_ch is a required parameter', func_);
end

%
% --- end of inner_check_arguments()

% --- END OF FILE --- %
