function [data] = vb_meg_yokogawa_store_bin_channel(ykgwfile, ...
  meg_info, out_base, verbose_swt)
% load measurement data and store it in a binary format for each channel
%
% [usage]
%   [data] = vb_meg_yokogawa_store_bin_channel(ykgwfile, ...
%     meg_info, out_base, verbose_swt)
%
% [input]
%      ykgwfile : <required> Yokogawa file
%      meg_info : <required> <<struct>> meg header information
%      out_base : <required> base directory of data directory
%               :            normally, it is the path of new MEG-MAT file.
%   verbose_swt : <optional> <<boolean>> switch to output verbose message [true]
%
% [output]
%          data : loaded measurement data [n_channel x n_sample x n_trial]
%
% [note]
%   <<prior conditions>>
%     the directory on which data will be stored has been already made.
%     MEGinfo.saveman.precision has set something value like 'float64'.
%
%   See also
%     vb_meg_yokogawa_store_bin
%
% [history]
%   2007-06-28 (Sako) initial version
%   2007-08-22 (Sako) changed stored data directory specification
%   2008-02-08 (Sako) modified according to change of channel spec
%   2009-06-23 (Sako) supported new data type : 'Evoked_Raw'
%   2009-07-13 (Sako) modiefied buffering ch_file, chf_id (second argument)
%   2011-02-17 (Sako) added new argument 'out_base'
%   2011-06-22 (Sako) modified to fit the new Yokogawa library
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('ykgwfile', 'var'), ykgwfile = ''; end
if ~exist('meg_info', 'var'), meg_info = []; end
if ~exist('out_base', 'var'), out_base = ''; end
if ~exist('verbose_swt', 'var'), verbose_swt = []; end
[ykgwfile, meg_info, out_base, VERBOSE] = ...
  inner_check_arguments(ykgwfile, meg_info, out_base, verbose_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
const = vb_define_yokogawa;
func_ = mfilename;

data = [];  % unused

% --- channel label list
meg_ch_str = vb_meginfo_get_channel_label_meg(meg_info);
ext_ch_str = vb_meginfo_get_channel_label_extra(meg_info);
ref_ch_str = vb_meginfo_get_channel_label_refmg(meg_info);
combined_ch_str = vb_meginfo_get_channel_label_whole(meg_info);

% --- NOTICE --- %
% As yokogawa channel name is made from numeric data,
% channel name re-convert to numeric data here for convenience.
Nch_meg = length(meg_ch_str);
meg_ch = zeros(Nch_meg,1);
for i_ch = 1:Nch_meg
  meg_ch(i_ch) = str2double(meg_ch_str(i_ch));
end

Nch_ext = length(ext_ch_str);
ext_ch = zeros(Nch_ext,1);
for i_ch = 1:Nch_ext
  ext_ch(i_ch) = str2double(ext_ch_str(i_ch));
end

Nch_ref = length(ref_ch_str);
ref_ch = zeros(Nch_ref,1);
for i_ch = 1:Nch_ref
  ref_ch(i_ch) = str2double(ref_ch_str(i_ch));
end

Nch = length(combined_ch_str);
combined_ch = zeros(Nch,1);
for i_ch = 1:Nch
  combined_ch(i_ch) = str2double(combined_ch_str(i_ch));
end

% --- number of channels
Nchannel_meg = vb_meginfo_get_channel_number_meg(meg_info);
Nchannel_ext = vb_meginfo_get_channel_number_extra(meg_info);
Nchannel_ref = vb_meginfo_get_channel_number_refmg(meg_info);
Nchannel_all = Nchannel_meg + Nchannel_ext + Nchannel_ref;

% --- number of sample
Nsample      = vb_meginfo_get_sample_number(meg_info);

% binary data type which will be used for fwrite
PRECISION = vb_meginfo_get_precision(meg_info);

% make buffers of file name
ch_file = cell(Nchannel_all,1);

% make file specifications
chf_fid = zeros(Nchannel_all,1);

% ---
% --- OPEN BINARY FILES FOR EACH CHANNEL
% --- 

data_dir = [out_base '/' vb_saveman_get_dir(meg_info.saveman)];

% --- CHANNEL
for ic = 1:Nchannel_all
  ch_file{ic} = sprintf('%s%s%d.%s', ...
    data_dir, filesep, combined_ch(ic), const.EXT_CHANNEL_BIN);
  
  chf_fid(ic) = fopen(ch_file{ic}, 'wb');
  if chf_fid(ic) == -1
    error('(%s)cannot open file (%s)', func_, ch_file{ic});
  end
end

if VERBOSE
  fprintf('opening channel file (%d) done.\n', Nchannel_all);
end

AD_label = 0:Nchannel_all-1;

%
% --- read Nblock samples and store for each channel
%

% maximum memory
MemoryMax = meg_info.memory_max;

Nblock = fix(MemoryMax/Nchannel_all);

acq_type = vb_meginfo_get_acqtype(meg_info);

% +1 is for channel name
block = zeros(Nchannel_all, Nblock+1);
% my_data =zeros(1,Nblock);

nstart = 0;

if VERBOSE
  digit_num = vb_util_get_digit_number(Nsample);
  MSG_LEN = inner_print_progress(nstart, Nsample, digit_num);
end

if strcmp(acq_type, 'Continuous_Raw') || strcmp(acq_type, 'Evoked_Ave')
  while nstart < Nsample
  	ndata = nstart + Nblock;
  	if ndata > Nsample
      if nstart == 0
        Nblock = Nsample;
      else
        Nblock = Nsample - nstart;
      end
    end

    block = getYkgwData(ykgwfile, nstart, Nblock);
    % --- (notice) return value of getYkgwData is calibrated yet.
    
    nstart = nstart + Nblock;
        
  	for ic = 1:Nchannel_all

      CUR_CH = combined_ch(ic);

      block_idx = CUR_CH == AD_label(:);
      my_data = block(block_idx,:);

      fwrite(chf_fid(ic), my_data(:), PRECISION);
    end

    % indicator
      if VERBOSE
        for ib = 1:MSG_LEN
          fprintf('\b');
        end
        inner_print_progress(nstart, Nsample, digit_num);
      end
  end

elseif strcmp(acq_type, 'Evoked_Raw')
  Ntrial = vb_info_get_trial_number(meg_info);
  
  if VERBOSE, n_line = 1; end

  % --- load 'Evoked_Raw' data by trial
  for i_trial = 1:Ntrial
    block = getYkgwData(ykgwfile, i_trial, 1);

    % --- store data by channel
    for i_ch = 1:Nchannel_all

      CUR_CH = combined_ch(i_ch);
      block_idx = CUR_CH == AD_label(:);

 		  my_data = block(block_idx,:);
  		fwrite(chf_fid(i_ch), my_data(:), PRECISION);
    end

    if VERBOSE
      fprintf('.');
      if (i_trial/50) == n_line
        fprintf('[%d/%d]\n', i_trial, Ntrial);
        n_line = n_line + 1;
      end
    end
  end % --- end of Ntrial

else
  error('(%s) unknown acq_type : %s', mfilename, acq_type);
end

if VERBOSE
  fprintf('\n');
end

for ic = 1:Nchannel_all
  fclose(chf_fid(ic));
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [ykgwfile, meg_info, out_base, verbose_swt] = ...
  inner_check_arguments(ykgwfile, meg_info, out_base, verbose_swt)

func_ = mfilename;

if isempty(ykgwfile)
  error('(%s) ykgwfile is a required parameter', func_);
end

if isempty(meg_info)
  error('(%s) meg_info is a required parameter', func_);
end

if isempty(out_base)
  error('(%s) out_base is a required parameter', func_);
end

if isempty(verbose_swt)
  verbose_swt = true;
end
return;
%
% --- end of inner_check_arguments()

% --- inner_print_progress()
%
function str_len = inner_print_progress(present_num, whole_num, digit_num)

cmd_tmp = sprintf('[%%0%dd / %%0%dd] (%%3d%%%%)', digit_num, digit_num);
prog_rate = floor((present_num / whole_num) * 100);
cmd_str = sprintf( ...
  'fprintf(''--- done : %s ...'', present_num, whole_num, prog_rate)', ...
    cmd_tmp);
eval(cmd_str);

% --- "--- done : [0000001 / 1234567] (  0%) ..."
%       4    7     5+digit_num*2         7    4
str_len = 4+7+5+digit_num*2+7+4;
return;
%
% --- end of inner_print_progress()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
