function [labels, result] = ...
  vb_meginfo_get_channel_label_extra(meginfo, active_swt, ref_swt)
% return label list of extra channel
% [usage]
%   [labels, result] = ...
%      vb_meginfo_get_channel_label_extra(meginfo, active_swt, ref_swt)
% [input]
%      meginfo : <required> <<struct>> MEGinfo which has 'ExtraChannelInfo'
%   active_swt : <optional> <<boolean>> active filter switch [false]
%              :   true) return active channels
%              :  false) return all the channels
%      ref_swt : <optional> <<boolean>> switch to include reference or not
%              :    true ) return list which includes reference channels
%              :  [false]) return list no-reference channels
% [output]
%       labels : label list of channels
%       result : <<integer>> error code
%              :  0) success
%              :  1) error - bad meginfo
% [note]
%   new ExtraChannelInfo
%     .Channel_active
%     .Channel_name
%     .Channel_type
%     .Channel_id
% [history]
%   2007-06-27 (Sako) initial version
%   2008-02-01 (Sako) modified according to new spec
%   2008-04-21 (Sako) added active_swt, ref_swt
%   2008-06-06 (Sako) modified by adding return condition
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('meginfo', 'var'), meginfo = []; end
if ~exist('active_swt', 'var'), active_swt = []; end
if ~exist('ref_swt', 'var'), ref_swt = []; end
[meginfo, active_swt, ref_swt, result] = ...
  inner_check_arguments(meginfo, active_swt, ref_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
labels = [];
if result ~= 0
  return;
end

labels  = vb_meginfo_get_channel_name_extra(meginfo, []);
types   = vb_meginfo_get_channel_type_extra(meginfo);
actives = vb_meginfo_get_channel_active_extra(meginfo, active_swt);

if isempty(labels) || isempty(types) || isempty(actives)
  return;
end

if length(labels) ~= length(types) ...
    || length(actives) ~= length(labels)
  fprintf('(%s) bad ExtraChannelInfo\n', mfilename);
  return;
end

% ----- channel type less than zero means extra channel
% ----- narrow by active filter and channel type
if ref_swt
  criterion = (actives == 1);
else
  % without reference
  criterion = bitand((actives == 1),(types <= 0));
end
labels = labels(criterion == 1);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [meginfo, active_swt, ref_swt, result] = ...
  inner_check_arguments(meginfo, active_swt, ref_swt)
func_ = mfilename;
result = 0;

if isempty(meginfo)
  fprintf('(%s)meginfo is a required parameter\n', func_);
  result = 1;
  return;
end

% meginfo must have EEGinfo field
if ~isfield(meginfo, 'ExtraChannelInfo')  || isempty(meginfo.ExtraChannelInfo)
  fprintf('(%s)meginfo must have ExtraChannelInfo field\n', func_);
  result = 1;
  return;
end

if isempty(active_swt)
  active_swt = false;
end

if isempty(ref_swt)
  ref_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
