function vb_megfile_make_new_megfile(orgfile, newfile, loadspec, ...
  meg_labels, ext_labels, ref_labels, meg, ext, ref, acqtype, new_samples)
% make new megfile
% [usage]
%   vb_megfile_make_new_megfile(orgfile, newfile, loadspec, ...
%     meg_labels, ext_labels, ref_labels, meg, ext, ref, acqtype, new_samples)
% [input]
%       orgfile : <required> <<file>> MEG-MAT file before loading
%       newfile : <required> <<file>> new MEG-MAT file which has loading data
%      loadspec : <required> <<struct>> this will be added to new MEG-MAT
%               :   as it is
%    meg_labels : <optional> {Nchannel_meg x 1} [[]]
%               :  label list of new meg channels
%    ext_labels : <optional> {Nchannel_ext x 1} [[]]
%               :  label list of new extra channels
%    ref_labels : <optional> {Nchannel_ref x 1} [[]]
%               :  label list of new refmg channels
%           meg : <optional> set MEG data [Nch_meg x Nsample x Ntrial] []
%               :  which will be stored to bexp field
%           ext : <optional> set EXTRA data [Nch_ext x Nsample x Ntrial] []
%               :  which will be stored to bexp_ext field
%           ref : <optional> set REFERENCE data [Nch_ref x Nsample x Ntrial]
%               :  [] which will be stored to refmg field
%       acqtype : <optional> <<string>> set acq_type field if this is set ['']
%   new_samples : <optional> from-to list [Ntrial x 2] or [1 x 2]
%               :  if loadspec.TrialNumber is empty,
%               :  MEGinfo.Trial is made by this.
% [output]
%   none
% [note]
%
% [history]
%   2008-02-21 (Sako) initial version
%   2008-04-24 (Sako) changed the rule of trial number continuous -> raw
%   2009-07-14 (Sako) added three label arguments (meg,ext,ref), modified Trial
%   2009-08-04 (Sako) modified to save only MEG channels in MEGinfo.ChannelInfo
%   2010-06-30 (Sako) supported storing data externally
%   2011-03-02 (Sako) added checking old style data fields exist or not
%   2011-07-07 (Sako) modified ParentInfo format
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('orgfile', 'var'), orgfile = []; end
if ~exist('loadspec', 'var'), loadspec = []; end
if ~exist('newfile', 'var'), newfile = []; end
if ~exist('meg_labels', 'var'), meg_labels = []; end
if ~exist('ext_labels', 'var'), ext_labels = []; end
if ~exist('ref_labels', 'var'), ref_labels = []; end
if ~exist('meg', 'var'), meg = []; end
if ~exist('ext', 'var'), ext = []; end
if ~exist('ref', 'var'), ref = []; end
if ~exist('acqtype', 'var'), acqtype = []; end
if ~exist('new_samples', 'var'), new_samples = []; end
[orgfile, newfile, LoadSpec, meg_labels, ext_labels, ref_labels, ...
  meg, ext, ref, acqtype, new_samples, data_save_ex, saveman] = ...
  inner_check_arguments(orgfile, newfile, loadspec, ...
    meg_labels, ext_labels, ref_labels, meg, ext, ref, acqtype, new_samples);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- make ParentInfo
tmp_org = load(orgfile);
if isfield(tmp_org, 'ParentInfo')
  ParentInfo.ParentInfo = tmp_org.ParentInfo;
end

if isfield(tmp_org, 'LoadSpec')
  ParentInfo.LoadSpec = tmp_org.LoadSpec;
end

% --- save parent file
ParentInfo.meg_file = orgfile;

clear tmp_org;

pick = [];
Qpick = [];
ref_pick = [];
ref_Qpick = [];
MEGinfo = [];
CoordType = '';
Measurement = '';
PositionFile = '';

% - load valid data
% --- pick
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'pick');
if state == const.VALID
  load(orgfile, 'pick');
end

% --- Qpick
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'Qpick');
if state == const.VALID
  load(orgfile, 'Qpick');
end

% --- ref_pick
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'ref_pick');
if state == const.VALID
  load(orgfile, 'ref_pick');
end

% --- ref_Qpick
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'ref_Qpick');
if state == const.VALID
  load(orgfile, 'ref_Qpick');
end

% --- MEGinfo
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'MEGinfo');
if state == const.VALID
  load(orgfile, 'MEGinfo');
end
ParentInfo.MEGinfo = MEGinfo;

% --- CoordType
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'CoordType');
if state == const.VALID
  load(orgfile, 'CoordType');
end

% --- Measurement
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'Measurement');
if state == const.VALID
  load(orgfile, 'Measurement');
end

% --- PositionFile
[state, const] = vb_util_check_variable_in_matfile(orgfile, 'PositionFile');
if state == const.VALID
  load(orgfile, 'PositionFile');
end

% --- new MEGinfo

% ----- acq_type
if ~isempty(acqtype)
  MEGinfo = vb_meginfo_set_acqtype(MEGinfo, acqtype);
end

Nchannel = 0;
Nsample  = 0;
Ntrial   = 0;

if ~isempty(meg)
  Nchannel = size(meg,1);
  Nsample  = size(meg,2);
  Ntrial   = size(meg,3);
elseif ~isempty(ext)
  Nsample = size(ext,2);
  Ntrial  = size(ext,3);
elseif ~isempty(ref)
  Nsample = size(ref,2);
  Ntrial  = size(ref,3);
end

% ----- Nchannel
MEGinfo = vb_meginfo_set_channel_number(MEGinfo, Nchannel);

% ----- Nsample
MEGinfo = vb_meginfo_set_sample_number(MEGinfo, Nsample);

% ----- Pretrigger
if ~isempty(loadspec.Pretrigger)
  MEGinfo = vb_meginfo_set_pre_trigger(MEGinfo, loadspec.Pretrigger);
else
  if ~isfield(MEGinfo, 'Pretrigger') || isempty(MEGinfo.Pretrigger)
    MEGinfo = vb_meginfo_set_pre_trigger(MEGinfo, 0);
  end
end

% ----- Nrepeat
MEGinfo = vb_info_set_Nrepeat(MEGinfo, Ntrial);

% ----- Trial
if ~isempty(loadspec.TrialNumber)
  % --- chopped trial --> narrowed down chopped trial
  old_trial = vb_info_get_trial_data(MEGinfo, loadspec.TrialNumber);

  sample_idx = new_samples(1,1):new_samples(1,2);

  for i_trial = 1:length(old_trial)
    new_trial(i_trial) = old_trial(i_trial);
    
    % --- update only sample
    sample_old = old_trial(i_trial).sample;
    new_trial(i_trial).sample = sample_old(sample_idx);
  end

else
  old_trial = vb_info_get_trial_data(MEGinfo);
  n_trial_old = length(old_trial);

  if n_trial_old == 1
    % --- continuous --> chopped trial or continuous
    n_trial_new = size(new_samples, 1);

    for i_trial = 1:n_trial_new
      sample_idx = new_samples(i_trial,1):new_samples(i_trial,2);
      new_trial(i_trial).sample = sample_idx;
      new_trial(i_trial).number = i_trial;
      new_trial(i_trial).Active = true;
    end
  else
    % --- chopped trial --> narrowed down chopped trial
    sample_idx = new_samples(1,1):new_samples(1,2);

    for i_trial = length(old_trial)
      new_trial(i_trial) = old_trial(i_trial);
    
      % --- update only sample
      sample_old = old_trial(i_trial).sample;
      new_trial(i_trial).sample = sample_old(sample_idx);
    end
  end
end
% [N x 1]
MEGinfo.Trial = vb_util_arrange_list(new_trial, 0);

MEGinfo = vb_info_adjust_trial(MEGinfo);

% --- MEG channel information
load_ch_name = vb_loadspec_get_channel_name(loadspec);

if ~isempty(load_ch_name)
  % ----- MEGch_id
  [ch_idx] = ...
    vb_meginfo_get_channel_index_meg(MEGinfo, load_ch_name);
  MEGinfo.MEGch_id = MEGinfo.MEGch_id(ch_idx);

% ----- MEGch_name
  MEGinfo.MEGch_name = MEGinfo.MEGch_name(ch_idx);
  
% ----- ActiveChannel
  active_ch = vb_info_get_active_channel(MEGinfo);
  if isempty(active_ch)
    Nch = length(MEGinfo.MEGch_id);
    MEGinfo = vb_info_set_active_channel(MEGinfo, ones(Nch,1));
  elseif length(active_ch) ~= length(MEGinfo.MEGch_id)
    MEGinfo = vb_info_set_active_channel(MEGinfo, active_ch(ch_idx));
  end

% ----- ChannelInfo
%   new_channel_info = vb_megfile_make_channel_info(orgfile, ...
%     meg_labels, ext_labels, ref_labels);
  % --- MEGinfo.ChannelInfo saves only MEG channels
  new_channel_info = vb_megfile_make_channel_info(orgfile, ...
    meg_labels, [], []);

else
  MEGinfo.MEGch_id = [];
  MEGinfo.MEGch_name = '';
  MEGinfo = vb_info_set_active_channel(MEGinfo, []);
  new_channel_info = [];
  ch_idx = [];
end

MEGinfo = vb_info_set_channel_info(MEGinfo, new_channel_info);

% ----- Sensor (pick,Qpick,sensor_weight)
if ~isempty(pick) && ~isempty(Qpick)
  [pick, Qpick, MEGinfo] = vb_meg_adjust_sensor(pick, Qpick, ch_idx, MEGinfo);
end

% ----- ExtraChannelInfo
if ~isempty(ref_pick) && ~isempty(ref_Qpick)
  [MEGinfo, ref_pick, ref_Qpick] = ...
    vb_meginfo_solve_extra_channel(MEGinfo, ref_pick, ref_Qpick, load_ch_name);
end


% ----- save sampling data
if ~data_save_ex
  bexp = meg;
  bexp_ext = ext;
  refmg = ref;
  MEGinfo.saveman = [];
  
else
  const = vb_define_yokogawa;
  data_ext = const.EXT_CHANNEL_BIN;
  
  bexp = [];
  bexp_ext = [];
  refmg = [];
  
  f_path = vb_get_file_parts(newfile);
  d_path = fullfile(f_path, vb_saveman_get_dir(saveman));

  prec = vb_saveman_get_precision(saveman);
  
  % --- MEG channel
  n_ch = length(meg_labels);
  for i_ch = 1:n_ch
    vb_util_make_external_data_file(d_path, meg_labels{i_ch}, ...
      data_ext, prec, meg(i_ch,:,:) );
  end
  
  % --- EXTRA channel
  n_ch = length(ext_labels);
  for i_ch = 1:n_ch
    vb_util_make_external_data_file(d_path, ext_labels{i_ch}, ...
      data_ext, prec, ext(i_ch,:,:) );
  end
  
  % --- REFERENCE channel
  n_ch = length(ref_labels);
  for i_ch = 1:n_ch
    vb_util_make_external_data_file(d_path, ref_labels{i_ch}, ...
      data_ext, prec, ref(i_ch,:,:) );
  end

  MEGinfo.saveman = saveman;
end

vb_fsave(newfile, 'pick', 'Qpick', 'ref_pick', 'ref_Qpick', 'CoordType', ...
  'bexp', 'bexp_ext', 'refmg', 'LoadSpec', 'ParentInfo', ...
  'MEGinfo', 'Measurement', 'PositionFile');
% end
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [orgfile, newfile, loadspec, meg_labels, ext_labels, ref_labels, ...
  meg, ext, ref, acqtype, new_samples, data_save_ex, saveman] = ...
  inner_check_arguments(orgfile, newfile, loadspec, ...
    meg_labels, ext_labels, ref_labels, meg, ext, ref, acqtype, new_samples)
func_ = mfilename;
if isempty(orgfile)
  error('(%s)orgfile is a required parameter', func_);
end

if exist(orgfile, 'file') ~= 2
  error('(%s)cannot find orgfile : %s', func_, orgfile);
end

if isempty(newfile)
  error('(%s)newfile is a required parameter', func_);
end
if isempty(loadspec)
  error('(%s)loadspec is a required parameter', func_);
end

% --- where data will be stored
data_save_ex = 0; % 0 means data will be stored internally (as before)
saveman = [];
if isfield(loadspec, 'saveman')
  saveman = loadspec.saveman;
  data_save_ex = vb_saveman_get_switch(loadspec.saveman);
end

if isempty(meg_labels)
  % require no action
end
if isempty(ext_labels)
  % require no action
end
if isempty(ref_labels)
  % require no action
end

if isempty(acqtype)
  % require no action
end
if isempty(new_samples)
  warning('(%s)new_samples is not specified\n', func_);
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
