function [position, direction, coilweight, center, result, channel_info] = ...
    vb_megfile_get_sensor_position_by_type(megfile, ch_type, active_swt)
% return information of sensor position
%
% [usage]
%   [position, direction, coilweight, center, result] = ...
%     vb_megfile_get_sensor_position(megfile, ch_type, active_swt)
%
% [input]
%      megfile : <required> <<file>> MEG-MAT file
%      ch_type : <optional> <<string>> user specifies sensor type ['']
%              :  '' or 'MEG' : MEG channels
%              :        'REF' : REFFERENCE channels
%              :        'ALL' : MEG and REFFERENCE channels
%              :      'AXIAL' : AXIAL channels
%              :     'PLANAR' : PLANAR channels
%              :  case-insensitive
%   active_swt : <optional> <<boolean>> [false] only for MEG channels
%              :    true) return only active channels (MEG)
%              :   false) return all the channels (MEG)
%
% [output]
%     position : position data of sensors
%              :  (pick in MEG-MAT) [Nchannel x 3]
%    direction : direction data of sensors
%              :  (Qpick in MEG-MAT) [Nchannel x 3]
%   coilweight : matrix of coil weight [Nchannel x Ncoil]
%              :  this value depends on ch_type (channel type) as follows
%              :   'MEG' : CoilWeight of MEG channels
%              :   'REF' : CoilWeight of REFFERENCE channels
%              :   'ALL' : CoilWeight of MEG and REFFERENCE channels
%       center : center coordinates [1 x 3]
%       resule : <<integer>> error code
%              :  0) success
%              :  1) error - bad megfile
%              :  2) error - bad Measurement
%              :  4) error - bad ch_type
% channel_info : <<struct>> channel information of loaded data
%                :  .Active [Nchannel x 1]
%                :  .Name   [Nchannel x 1]
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
%                empty return : including reference sensor name case('REF', 'ALL').
% [note]
%   See also
%     vb_load_yokogawa_sensor
%
% [history]
%   2007-07-24 (Sako) initial version
%   2008-03-31 (Sako) supported the case of empty Vcenter
%   2008-04-15 (Sako) added active_swt, result
%   2009-09-01 (Sako) replaced active cheking function
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%   2011-07-26 (Sako) added two types channels as supported 'AXIAL' and 'PLANAR'
%   2011-08-31 (Sako) modified how to get data in the case of unknown device
%   2017-03-06 (rhayashi) Add channel_info.
%   2018-08-23 (Takeda) supported NEUROMAG
%   2022-06-03 (k_suzuki) supported QZFM
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = []; end
if ~exist('ch_type', 'var'), ch_type = ''; end
if ~exist('active_swt', 'var'), active_swt = []; end
[megfile, ch_type, active_swt, result] = ...
    inner_check_arguments(megfile, ch_type, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
position = [];
direction = [];
coilweight = [];
center = [];
channel_info = [];
if result ~= 0
    return;
end

func_ = mfilename;

[Measurement, device] = vb_load_device(megfile);

Device = upper(device);

if ~strcmpi(Measurement, 'MEG')
    msg = sprintf('(%s) Measurement is ''%s'', not ''MEG''.(%s)', ...
        func_, Measurement, megfile);
    vb_disp(msg);
    result = 2;
    return;
end

switch  Device
    case  'SBI'
        [position, direction, coilweight, center] = ...
            inner_get_info_sbi(megfile, ch_type);
        
    case  'YOKOGAWA'
        [position, direction, coilweight, center, channel_info] = ...
            inner_get_info_yokogawa(megfile, ch_type, active_swt);
        
    case 'NEUROMAG'
        [position, direction, coilweight, center, channel_info] = ...
            inner_get_info_neuromag(megfile, ch_type, active_swt);
        
    case 'QZFM'
        [position, direction, coilweight, center, channel_info] = ...
            inner_get_info_qzfm(megfile, ch_type, active_swt);
        
    otherwise
        msg = sprintf('(%s)unexpected Device : %s', func_, Device);
        vb_disp(msg);
        
        result = 2;
        [position, direction, coilweight, center] = ...
            inner_get_info_yokogawa(megfile, ch_type, active_swt);
end

% arrange center to [1 x 3]
if ~isempty(center)
    center = vb_util_arrange_list(center, 1);
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, ch_type, active_swt, result] = ...
    inner_check_arguments(megfile, ch_type, active_swt)
func_ = mfilename;
result = 0;

if isempty(megfile)
    msg = sprintf('(%s)MEG-MAT file is a required parameter', func_);
    vb_disp(msg);
    %   fprintf('(%s)MEG-MAT file is a required parameter\n', func_);
    result = 1; % error code 1 means bad megfile
    return;
end

if exist(megfile, 'file') ~= 2
    msg = sprintf('(%s)cannot find MEG-MAT file : %s', megfile);
    vb_disp(msg);
    %   fprintf('(%s)cannot find MEG-MAT file : %s\n', megfile);
    result = 1;
    return;
end

if isempty(ch_type)
    ch_type = 'MEG';
end

% case-insensitive
ch_type = upper(ch_type);

if isempty(active_swt)
    active_swt = false;
end
return;
%
% --- end of inner_check_arguments()


% --- inner_get_info_sbi
%
function [position, direction, coilweight, center] = ...
    inner_get_info_sbi(megfile, ch_type)

[position, direction, coilweight, center] = load_sbi_sensor(megfile);

switch ch_type
    case  'MEG'
        % require no special action
    case  'REF'
        position = [];
        direction = [];
        coilweight = [];
    case  'ALL'
        % require no special action
    otherwise
        error('(%s) undefined ch_type : %s', mfilename, ch_type);
end
return;
%
% --- end of inner_get_info_sbi()

% --- inner_get_info_yokogawa
%
function [position, direction, coilweight, center, channel_info] = ...
    inner_get_info_yokogawa(megfile, ch_type, active_swt)

channel_info = [];

switch ch_type
    case  'MEG'
        % get MEG channel information
        [pick, Qpick, CW, center, channel_info] = ...
            vb_load_yokogawa_sensor(megfile, false, active_swt);
        
        position = pick;
        direction = Qpick;
        coilweight = CW;
        
    case  'REF'
        % get REFFERENCE channel information
        [ref_pick, ref_Qpick, ref_CW, center] = ...
            vb_load_yokogawa_sensor(megfile, true, active_swt);
        
        position = ref_pick;
        direction = ref_Qpick;
        coilweight = ref_CW;
        
    case  'ALL'
        % get MEG channel information
        [pick, Qpick, CW, center] = ...
            vb_load_yokogawa_sensor(megfile, false, active_swt);
        
        % get REFFERENCE channel information
        [ref_pick, ref_Qpick, ref_CW, center] = ...
            vb_load_yokogawa_sensor(megfile, true, active_swt);
        
        position = [pick; ref_pick];
        direction = [Qpick; ref_Qpick];
        
        meg_ch_len = size(CW,1);
        meg_coil_len = size(CW,2);
        ref_ch_len = size(ref_CW,1);
        ref_coil_len = size(ref_CW,2);
        
        ch_len = meg_ch_len + ref_ch_len;
        coil_len = meg_coil_len + ref_coil_len;
        
        % initialize
        coilweight = zeros(ch_len, coil_len);
        
        % MEG channel
        coilweight(1:meg_ch_len, 1:meg_coil_len) = CW;
        coilweight((meg_ch_len+1):ch_len, (meg_coil_len+1):coil_len) = ref_CW;
        
    otherwise
        % ---  supporse that a type name is specified
        % ---  ONLY 'MEG' CHANNELS ('AXIAL' or 'PLANAR')
        ch_info = vb_load_channel_info(megfile, ch_type);
        if isempty(ch_info)
            error('(%s) unknown ch_type : %s', mfilename, ch_type);
        end
        
        [position, direction, coilweight, center, channel_info] = ...
            vb_megfile_get_sensor_position_by_name(megfile, ch_info.Name, active_swt);
end

% --- solve center
if isempty(center)
    meginfo = vb_megfile_load_meginfo(megfile);
    
    if isfield(meginfo, 'Vcenter')
        % center coordinate
        center = meginfo.Vcenter;
    end
end


return;
%
% --- end of inner_get_info_yokogawa()

% --- inner_get_info_neuromag
%
function [position, direction, coilweight, center, channel_info] = ...
    inner_get_info_neuromag(megfile, ch_type, active_swt)

% get MEG channel information
[position, direction, coilweight, center, channel_info] = ...
    vb_load_neuromag_sensor(megfile, active_swt);

switch ch_type
    case  'MEG'
        % require no special action
    case  'REF'
        position = [];
        direction = [];
        coilweight = [];
    case  'ALL'
        % require no special action
    otherwise
        % ---  supporse that a type name is specified
        % ---  ONLY 'MEG' CHANNELS ('MAG' or 'AXIAL' or 'PLANAR')
        ch_info = vb_load_channel_info(megfile, ch_type);
        if isempty(ch_info)
            error('(%s) unknown ch_type : %s', mfilename, ch_type);
        end
        
        [position, direction, coilweight, center, channel_info] = ...
            vb_megfile_get_sensor_position_by_name(megfile, ch_info.Name, active_swt);
end
return;
%
% --- end of inner_get_info_neuromag()

% --- inner_get_info_qzfm
%
function [position, direction, coilweight, center, channel_info] = ...
    inner_get_info_qzfm(megfile, ch_type, active_swt)

% get MEG channel information
[position, direction, coilweight, center, channel_info] = ...
    vb_load_neuromag_sensor(megfile, active_swt);

switch ch_type
    case  'MEG'
        % require no special action
    case  'REF'
        position = [];
        direction = [];
        coilweight = [];
    case  'ALL'
        % require no special action
    otherwise
        % ---  supporse that a type name is specified
        % ---  ONLY 'MEG' CHANNELS ('MAG_X' or 'MAG_Y' or 'MAG_Z')
        ch_info = vb_load_channel_info(megfile, ch_type);
        if isempty(ch_info)
            error('(%s) unknown ch_type : %s', mfilename, ch_type);
        end
        if isempty(ch_info.ID)
            error('There is no data with ch_type %s', ch_type);
        end
        
        [position, direction, coilweight, center, channel_info] = ...
            vb_megfile_get_sensor_position_by_name(megfile, ch_info.Name, active_swt);
end
return;
%
% --- end of inner_get_info_qzfm()

%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
