function vb_eegfile_update_data(inst_spec)
% Replace data and update EEG-MAT file.
%
% This function is used to create/update EEG-MAT file with arbitrary
% signal preprocessing function not supported by VBMEG. 
%
% [syntax]
%   vb_eegfile_update_data(inst_spec)
%
% [input]
%   inst_spec : <<required>> <struct> instruction spec
%         :  fields are as follows:
%         :   .org_file     : <<required>> EEG-MAT file which will be updated
%         :   .new_data     : <<required>> [n_channel x n_sample]
%         :   .channel_type : <<optional>> ['EEG'] | 'EXTRA'
%         :   .new_file     : <<optional>> if you want to make new file.
%         :                 :  ['org_file']
%         :                 :  Empty means overwriting '.org_file'.
%         :   .bin_data_dir : <<optional>> data stored directory.
%         :                 : This parameter is valid only when 
%         :                 : '.new_file' field is set.
%         :                 : [(body of 'new_file')_bin]
%         :                 : e.g. './new_0123.eeg.mat' --> './new_0123_bin'
%
% [output]
%   void
% [note]
% 
% [history]
%   2011-02-08 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('inst_spec', 'var'), inst_spec = []; end
[inst_spec] = inner_check_arguments(inst_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
org_file = inst_spec.org_file;
[ch_info] = vb_load_channel_info(org_file, inst_spec.channel_type);


if ~isempty(inst_spec.new_file)
  % --- copy org_file - by using vb_load_meg_data()

  load_spec.ChannelType = 'ALL';
  load_spec.bin_data_dir = inst_spec.bin_data_dir;
  vb_load_meg_data(org_file, load_spec, inst_spec.new_file);


else
  % --- overwrite original file
  inst_spec.new_file = org_file;
end


% ------------------- %
% --- UPDATE DATA --- %
% ------------------- %
eeg_info = vb_load_measurement_info(inst_spec.new_file);
tmp_dir = vb_eeginfo_get_datadir(eeg_info);
eeg_root = vb_get_file_parts(inst_spec.new_file);

data_dir = [eeg_root '/' tmp_dir];

PRECISION = vb_eeginfo_get_datatype(eeg_info);
% --- PRECISION [N x 1]

vb_define_device;

n_channel = size(ch_info.Name, 1);
for i_ch = 1:n_channel
  ch_name = ch_info.Name{i_ch};
  cur_ch_file = sprintf('%s/%s.%s', ...
    data_dir, ch_name, FILE_EXT_BIN_CH_EEG);

  cur_fid = fopen(cur_ch_file, 'wb');
  if cur_fid == -1
    error('(%s)cannot open file (%s)', mfilename, cur_ch_file);
  end

  fwrite(cur_fid, inst_spec.new_data(i_ch, :), PRECISION{i_ch});
  fclose(cur_fid);
%   fprintf('--- UPDATED %s\n', cur_ch_file);
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [inst_spec] = inner_check_arguments(inst_spec)
if isempty(inst_spec)
  error('(%s) inst_spec is a required parameter', mfilename);
end

if ~isfield(inst_spec, 'org_file') || isempty(inst_spec.org_file)
  error('(%s) inst_spec.org_file is a required field', mfilename);
end

if exist(inst_spec.org_file, 'file') ~= 2
  error('(%s) cannot find inst_spec.org_file : %s', ...
    mfilename, inst_spec.org_file);
end

if ~isfield(inst_spec, 'new_data') || isempty(inst_spec.new_data)
  error('(%s) inst_spec.new_data is a required field', mfilename);
end

if ~isfield(inst_spec, 'channel_type') || isempty(inst_spec.channel_type)
  inst_spec.channel_type = 'EEG';
end

if ~isfield(inst_spec, 'new_file')
  inst_spec.new_file = '';

elseif ~isfield(inst_spec, 'bin_data_dir')
  inst_spec.bin_data_dir = '';
end
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
