function result = vb_eegfile_trial_ch_data(proc_spec)
% Make trial data files after extracting trials from continuous raw data
% [usage]
%   result = vb_eegfile_trial_ch_data(proc_spec)
% [input]
%   proc_spec : <required> <<struct>> struct defined process specifications
%             :  .root_dir  : data directory for new eeg file and trig_file
%             :             : <optional> ['.'] (current directory)
%             :  .trig_file : <required> <<file>> trigger file name
%             :             :  this file is expected to have data as follows:
%             :             :   .trig [1 x Ntrial]
%             :             :   .parm.data_file (without path)
%             :             :   .parm.Pretrigger_ms
%             :             :   .parm.Posttrigger_ms
%             :  .new_file  : new EEG filename
%             :             :  ['<root_dir>/
%             :             :    <old_meg_file>_epoch_yymmdd-HHMMSS'.eeg.mat]
%             :  .proc_parm :  <<struct>>
%             :             :  .Pretrigger_ms  : Pretrigger period   [msec]
%             :             :  .Posttrigger_ms : Posttrigger period  [msec]
%                              if they are specified, 
%                              they overwrite the setting values of trig_file
% [output]
%      result : <<struct>> depends on current process
% [note]
%
% [history]
%   2009-07-23 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('proc_spec', 'var'), proc_spec = []; end
[proc_spec, trig_file, data_file] = inner_check_arguments(proc_spec);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
result = [];

trig = load(trig_file);

% --- pretrigger / posttrigger
if ~isempty(proc_spec.proc_parm)
  if isfield(proc_spec.proc_parm, 'Pretrigger_ms') ...
    && ~isempty(proc_spec.proc_parm.Pretrigger_ms)
    pretrigger_ms = proc_spec.proc_parm.Pretrigger_ms;
  else
    pretrigger_ms = trig.parm.Pretrigger_ms;
  end
  
  if isfield(proc_spec.proc_parm, 'Posttrigger_ms') ...
    && ~isempty(proc_spec.proc_parm.Posttrigger_ms)
    posttrigger_ms = proc_spec.proc_parm.Posttrigger_ms;
  else
    posttrigger_ms = trig.parm.Posttrigger_ms;
  end
else
  pretrigger_ms  = trig.parm.Pretrigger_ms;
  posttrigger_ms = trig.parm.Posttrigger_ms;
end

load_spec.Trigger = trig.trig;

% convert trigger data from msec to sample
load_spec.Pretrigger  = ceil(pretrigger_ms *(fsamp/1000));
load_spec.Posttrigger = ceil(posttrigger_ms *(fsamp/1000));

load_spec.ChannelType = 'ALL';
new_file = proc_spec.new_file;

[eeg_data, ch_info] = vb_load_meg_data(data_file, load_spec, new_file);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [proc_spec, trig_file, data_file] = inner_check_arguments(proc_spec)
func_ = mfilename;

if isempty(proc_spec)
  error('(%s) proc_spec is a required parameter', func_);
end

% --- proc_spec
%  .root_dir
%  .trig_file
%  .new_file
%  .proc_parm
if ~isfield(proc_spec, 'root_dir') || isempty(proc_spec.root_dir)
  proc_spec.root_dir = '.';
end

if ~isfield(proc_spec, 'proc_parm') 
  proc_spec.proc_parm = [];
end

% --- check trigger file
if ~isfield(proc_spec, 'trig_file') || isempty(proc_spec.trig_file)
  error('(%s) trig_file is a required field of proc_spec', func_);
end

trig_file = [proc_spec.root_dir '/' proc_spec.trig_file];

if exist(trig_file, 'file') ~= 2
  error('(%s) cannot find proc_spec.trig_file : %s', ...
    func_, trig_file);
end

trig = load(trig_file);
if ~isfield(trig, 'trig') || isempty(trig.trig)
  error('(%s) trig_file must have trig field', func_);
end

if isfield(trig, 'parm') && ~isempty(trig.parm)
  if isfield(trig.parm, 'data_file') && ~isempty(trig.parm.data_file)
    data_file = [proc_spec.root_dir '/' trig.parm.data_file];
  else
    error('(%s) trig_file.parm must have data_file field', func_);
  end
else
  error('(%s) trig_file must have parm field', func_);
end

if exist(data_file, 'file') ~= 2
  error('(%s) cannot find data_file : %s', func_, data_file);
end

% --- new eeg file
if ~isfield(proc_spec, 'new_file') || isempty(proc_spec.new_file)
  [fpath, fname] = vb_get_file_parts(data_file);
  proc_spec.new_file = ...
    sprintf('%s_epoch_%s.eeg.mat', fname, datestr(now, 'yymmdd-HHMMSS'));
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
