function my_spec = vb_util_make_mark_trigger_spec( ...
  msrmnt_file, target_list, to_read_swt, output_file, auto_info)
% make struct of mark_trigger_spec from given information
%
% [usage]
%   my_spec = vb_util_make_mark_trigger_spec( ...
%     msrmnt_file, target_list, to_read_swt, output_file, auto_info)
%
% [input]
%   msrmnt_file : <required>
%               : MAT file which includes next information :
%               :   - data (Nchannel x Nsample) <must>
%               :   - base channel list <must>
%               :   - sample_frequency <if needed>
%               : e.g. EEG-MAT (.eeg.mat), MEG-MAT (.meg.mat)
%   target_list : <optional> [[]]
%               : list of used channel label
%               :   "base channel list" must have these channels
%   to_read_swt : <optional> true or false [false]
%               :   switch that "target_channel_list" is whether to read
%               :   or not
%               :   error if "to_read_swt" is true and "target_list" is empty
%   output_file : <optional> ['./trigger_list.mat']
%     auto_info : <optional> <<struct>> [[]]
%         : information for auto marking. fields are as follows:
%         :   - auto_detect
%         :     : true or false
%         :     : auto-detect switch
%         :     : if this switch is 'true', other fields of auto_info are
%         :     : ignored.
%         :   - trigger_channel
%         :     : trigger channel label of "base channel list"
%         :   - trigger_file
%         :     : MAT file which includes trigger time list
%
% [output]
%       my_spec : <<struct>>
%         :   mark_trigger_spec type whose fields are as follows:
%         :                 data : <required> (Nchannel x Nsample)
%         :                      :  measured data matrix
%         :             plot_idx : <required> (Nchannel x 1)
%         :                      :  list of index of 'data' to be plotted
%         :  trigger_channel_idx : <optional> (Nchannel x 1) [[]]
%         :                      :  trigger index of 'data'
%         :   trigger_sample_idx : <optional> [[]]
%         :                      :  list of trigger points as sample number
%         :          output_file : <optional> ['.\trigger_(date).mat']
%         :                      :  output file which has trigger_sample_list
%
% [note]
%   
% [history]
%   2007-01-10 (Sako) initial version
%   2011-07-29 (Sako) replaced searching index function
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('msrmnt_file', 'var'), msrmnt_file = []; end;
if ~exist('target_list', 'var'), target_list = []; end;
if ~exist('to_read_swt', 'var'), to_read_swt = []; end;
if ~exist('output_file', 'var'), output_file = []; end;
if ~exist('auto_info', 'var'),   auto_info = [];   end;

[msrmnt_file, target_list, to_read_swt, output_file, auto_info] = ...
  inner_check_arguments( ...
  msrmnt_file, target_list, to_read_swt, output_file, auto_info);

% --- MAIN PROCEDURE ----------------------------------------------------------%
%
my_spec = [];

% --- 1) measured data
my_spec.data     = vb_load_meg_data(msrmnt_file);

base_channel_list = vb_msrmnt_get_channel_list(msrmnt_file);

read_channel_list = vb_util_get_labels_to_read( ...
  base_channel_list, target_list, to_read_swt);

% --- 2) plot index list
my_spec.plot_idx = ...
  vb_util_get_matching_label_index(base_channel_list, read_channel_list);

% --- 3) 4) trigger information (indices of channel and sample)
sampling_frequency = vb_msrmnt_get_sampling_frequency(msrmnt_file);

[trigger_channel_idx, trigger_sample_idx] = ...
  inner_get_trigger_info(msrmnt_file, auto_info, sampling_frequency);
% [trigger_channel, trigger_list] = inner_get_trigger_info( ...
%   msrmnt_file, auto_info, sampling_frequency);

my_spec.trigger_sample_idx = trigger_sample_idx;
my_spec.trigger_channel_idx = trigger_channel_idx;

% --- 5) output file
my_spec.output_file = output_file;
%
% --- END OF MAIN PROCEDURE ---------------------------------------------------%


% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments() -------------------- %
%
function [msrmnt_file,target_list,to_read_swt,output_file,auto_info] = ...
  inner_check_arguments(msrmnt_file, ...
    target_list, to_read_swt, output_file, auto_info)

vb_define_device;

% required parameter
if isempty(msrmnt_file)
  error('msrmnt_file is a required parameter');
end

if exist(msrmnt_file, 'file') ~= 2
  error('cannot read measurement file : %s', msrmnt_file);
end

% optional parameters
if isempty(to_read_swt)
  to_read_swt = false;
end

if isempty(output_file)
  output_file = ['.' filesep 'trigger_list.mat'];
end

% error if to_read_swt is true and target_list is empty
if isempty(target_list) && to_read_swt == true
  error('target_list is empty and to_read_swt is true');
end

% how to get initial triggers
if isfield(auto_info, 'auto_detect') ...
    && auto_info.auto_detect == true
  auto_info.how = TRIGGER_HOW_FULL;
  
elseif isfield(auto_info, 'trigger_channel') ...
    && ~isempty(auto_info.trigger_channel)
  auto_info.how = TRIGGER_HOW_SEMI;
  
elseif isfield(auto_info, 'trigger_file') ...
    && ~isempty(auto_info.trigger_file)
  auto_info.how = TRIGGER_HOW_FILE;
  
else
  auto_info.how = TRIGGER_HOW_MANUAL;
end
%
% --- end of inner_check_argument() -------------------- %


% --- inner_get_trigger_info() -------------------- %
% 
% function [trigger_channel, trigger_list] = inner_get_trigger_info( ...
%   auto_info, sampling_frequency, bit_list)
function [trigger_channel_idx, trigger_sample_idx] = ...
  inner_get_trigger_info(msrmnt_file, auto_info, sampling_frequency)

vb_define_device;

trigger_channel_idx = [];
trigger_sample_idx = [];

if isempty(auto_info)
  return;
end

% check highest priority switch
switch auto_info.how
  case  TRIGGER_HOW_FULL

    % auto-detect trigger list
    [trigger_channel_idx, trigger_sample_idx] = ...
      vb_msrmnt_auto_detect_trigger_info(msrmnt_file);
    return;

  case  TRIGGER_HOW_SEMI
    trigger_channels = auto_info.trigger_channel;

    [trigger_channel_idx, trigger_sample_idx] = vb_msrmnt_get_trigger_samples( ...
      msrmnt_file, trigger_channels);
    return;

  case  TRIGGER_HOW_FILE
    [trigger_channel_idx, trigger_sample_idx] = ...
      vb_util_read_trigger_file(auto_info.trigger_file, sampling_frequency);
  
  otherwise
    % nothing to do
end % end of switch auto_spec.how
%
% --- end of inner_get_trigger_info() -------------------- %

%%% END OF FILE %%%
