function [eeg_light_info] = vb_eeg_make_light_info(bdf_spec)
% make eeg_light_info struct from acquisition files
% [usage]
%   [eeg_light_info] = vb_eeg_make_light_info(bdf_spec)
% [input]
%   bdf_spec : <required> <<struct>> light version of bdf_spec struct
%            :  .bdf_file   <required> *.BDF file
%            :  .digit_file <required> *.digit file
%            :  .face_file  <required> *.face file
% [output]
%   eeg_light_info : <struct>
%     : Measurement           : measurement type 'EEG'
%     : header                : Biosemi header informatons
%     : ChannelLabel_hardware : channel label list of hardware
%     : ChannelLabel_standard : standard channel label list
%     : Nchannel              : number of channel of original data
%     : SampleFrequency       : sample rate of original data [Hz]
%     : RecordTime            : record time of original data [sec]
%     : Nsample               : total number of sample
%     :                               :  SampleFrequency * RecordTime
%     : Coord                 : position coordinates of sensor [n_sensor x 3]
%     : CoodType              : coordinates' information
% [note]
% 
% [history]
%   2007-08-31 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
[bdf_file, digit_file, face_file, Vcenter] = inner_read_bdf_spec(bdf_spec);


% --- MAIN PROCEDURE --------------------------------------------------------- %
%
[header, RecordTime, Nch, SampleRate, ChannelLabel_hardware] = ...
  inner_get_local_var_bdf_file(bdf_file);

[SensorPosition, CoordType, ChannelLabel_standard] ...
  = inner_get_info_digit_file(digit_file);
[Vcenter] = inner_get_info_face_file(face_file)


eeg_light_info.measurement           = 'EEG';
eeg_light_info.version               = 'LIGHT'
eeg_light_info.header                = header;
eeg_light_info.device                = 'BIOSEMI';
eeg_light_info.ChannelLabel_hardware = ChannelLabel_hardware;
eeg_light_info.Nchannel              = Nch;
eeg_light_info.SampleFrequency       = SampleRate;
eeg_light_info.RecordTime            = RecordTime;
eeg_light_info.Nsample               = RecordTime * SampleRate;
eeg_light_info.ChannelLabel_standard = ChannelLabel_standard;
eeg_light_info = vb_eeginfo_set_sensor_position(eeg_light_info, SensorPosition);
eeg_light_info.CoordType             = CoordType;
eeg_light_info.Vcenter               = Vcenter;

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_read_bdf_spec()
%
function [bdf_file, digit_file, face_file, Vcenter] = ...
  inner_read_bdf_spec(bdf_spec)

vb_define_device;

func_ = mfilename;

%
% --- BDF FILE
%
if ~isfield(bdf_spec, 'bdf_file') || isempty(bdf_spec.bdf_file)
  error('(%s)bdf_file is a required parameter of bdf_spec', func_);
end

bdf_file = bdf_spec.bdf_file;

if exist(bdf_file, 'file') ~= 2
  error('(%s)cannot read bdf_file : %s', func_, bdf_file);
end

% 
% --- DIGIT FILE
%
if ~isfield(bdf_spec, 'digit_file') || isempty(bdf_spec.digit_file)
  error('(%s)digit_file is a required parameter of bdf_spec', func_);
end

digit_file = bdf_spec.digit_file;

if exist(digit_file, 'file') ~= 2
  error('(%s)cannot read digit_file : %s', func_, digit_file);
end

% 
% --- FACE FILE
%
Vcenter = [];
face_file = [];
if ~isfield(bdf_spec, 'face_file') || isempty(bdf_spec.face_file)
  Vcenter = [0 0 0];
else
  face_file = bdf_spec.face_file;

  if exist(digit_file, 'file') ~= 2
    warning('(%s)cannot read digit_file : %s', func_, digit_file);
    Vcenter = [0 0 0];
  end
end
%
% --- end of inner_check_argument()

% --- inner_get_local_var_bdf_file()
%
function [header, RecordTime, Nch, SampleRate, ChannelLabel_hardware] = ...
  inner_get_local_var_bdf_file(bdf_file)

[DAT,H1] = vb_openbdf(bdf_file);

% procedure parameters
header     = DAT.Head;
RecordTime = DAT.Head.NRec;
Nch        = DAT.Head.NS;
SampleRate = max(DAT.Head.SampleRate);

ChannelLabel_hardware = cell(1,Nch);

% get channel label for hardware
for ch = 1:Nch
  ChannelLabel_hardware{ch} = deblank(DAT.Head.Label(ch,:));
end
%
% --- end of inner_get_local_var_bdf_file()

% --- inner_get_info_digit_file()
%
function [SensorPosition, CoordType, ChannelLabel] ...
  = inner_get_info_digit_file(digit_file)

eeg_sensor = [];
SensorPosition = [];
CoordType = [];
ChannelLabel = [];

load(digit_file, 'eeg_sensor');

if isempty(eeg_sensor)
  error('invalid digit_file:%s - cannot find eeg_sensor field', digit_file);
end;
if ~isfield(eeg_sensor, 'mri')
  error('invalid digit_file:%s - cannot find ''mri'' field', digit_file);
end
if ~isfield(eeg_sensor, 'name') 
  error('invalid digit_file:%s - cannot find ''name'' field', digit_file);
end
if ~isfield(eeg_sensor, 'coord_type_mri')
  error('invalid digit_file:%s - cannot find ''coord_type_mri'' field', digit_file);
end
  
SensorPosition = eeg_sensor.mri;
CoordType = eeg_sensor.coord_type_mri;
  
% eeg_sensor.name is [n_channel x 1];
ChannelLabel = vb_util_arrange_list(eeg_sensor.name);
return;
%
% --- end of inner_get_info_digit_file()

% --- inner_get_info_face_file()
%
function [Vcenter] = inner_get_info_face_file(face_file)
subject = [];
load(face_file, 'subject');
if isempty(subject)
  error('invalid face_file:%s - cannot find subject field', face_file);
end
if ~isfield(subject, 'spherical_head')
  error('invalid face_file:%s - cannot find spherical_head field', face_file);
end
  
if ~isfield(subject.spherical_head, 'Vcenter')
  error('invalid face_file:%s - cannot find Vcenter field', face_file);
end
  
Vcenter = subject.spherical_head.Vcenter;
return;
%
% --- end of inner_get_info_face_file()

%%% END OF FILE %%%
