function [data, channel_info, cond_id] = vb_load_active_trial_data(fileinfo)
% load meg/eeg data from multiple files and concatenate them
% [Usage]
%   [data, channel_info, cond_id] = vb_load_active_trial_data(fileinfo)
%
% [Input]
% fileinfo : structure with file information for multiple files
% fileinfo.filename{n}  : n-th session data file names [n=1:Nsession]
% fileinfo.Nchannel     : # of total channels
% fileinfo.Nsample      : # of samples in one trial
% fileinfo.Ntotal       : # of all trials 
% fileinfo.Ntrial       : # of trials for each session [1 x Nsession]
% fileinfo.session_id   : session index for each trial [1 x Ntotal]
% fileinfo.cond_id      : condition number for each trials [1 x Ntotal]
% fileinfo.ActiveChannel: Active channel list (boolean array) [Nchannel x 1]
% fileinfo.ActiveTrial  : Active trial list   (boolean array) [Ntotal x 1]
%
% [Output]
%         data : concatenated data data [Nch x Nsample x Nactive]
% channel_info : <<struct>> channel information of loaded data
%              :  .Active [Nchannel x 1]
%              :  .Name   [Nchannel x 1]
%              :  .Type   [Nchannel x 1]
%              :  .ID     [Nchannel x 1]
%      cond_id : condition number for each trials [1 x Nactive]
%
% Masa-aki Sato 2009-8-4
%   2010-11-15 (Sako) modified loading data of each file MEG --> ALL
%   2011-03-01 (Sako) supported empty channel_id (return empty data)
%   2011-08-26 (Sako) modified for minimum format file
%   2018-08-22 (Takeda) Neuromag is supported
%   2022-06-03 (k_suzuki) QZFM is supported
%   2022-06-03 (k_suzuki) To load multi-run from .info.mat with specified ActiveTrial, loadspec.ActiveTrial is set to false (line 107)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

filename  = fileinfo.filename  ;

Nsample  = fileinfo.Nsample;
Nchannel = fileinfo.Nchannel;
Ntotal   = fileinfo.Ntotal;

% Number of files
Nfile = length(filename);

% active channel & trials
if isfield(fileinfo, 'act_trial')
    act_trial = fileinfo.act_trial;
elseif isfield(fileinfo, 'ActiveTrial');
    act_trial = find(fileinfo.ActiveTrial ~= 0);
end

if isfield(fileinfo, 'channel_id');
    channel_id = fileinfo.channel_id;
elseif isfield(fileinfo, 'ActiveChannel')
    ch_info = vb_load_channel_info(filename{1});
    channel_id = ch_info.ID(fileinfo.ActiveChannel ~= 0);
end

if isempty(channel_id)
  warning('(%s) channel_id is empty\n', mfilename);
  data = [];
  channel_info = [];
  cond_id = [];
  return;
end

% --- get the name list of target channel
ch_info = vb_load_channel_info(filename{1}, 'USER', channel_id);
ch_names = ch_info.Name;

% Number of active channel & trials
Nch  = length(channel_id);
Nall = length(act_trial);

fprintf('Number of files to read: %d\n',Nfile)
fprintf('Number of all trials: %d\n',Ntotal)
fprintf('Number of all channels: %d\n',Nchannel)
fprintf('Number of trials to read: %d\n',Nall)
fprintf('Number of channels to read: %d\n',Nch)

data = zeros(Nch,Nsample,Ntotal);
Nst = 0;

channel_info = [];

% Merge multiple data files
for n=1:Nfile
  [measurement, device] = vb_load_device(filename{n});

  Measurement = upper(measurement);
  Device      = upper(device);

  % Load MEG/EEG data
  if (strcmp(Measurement, 'MEG') ...
    && (~strcmp(Device, 'YOKOGAWA') && ~strcmp(Device, 'BASIC') && ~strcmp(Device, 'NEUROMAG') && ~strcmp(Device, 'QZFM')))...
    || (strcmp(Measurement, 'EEG') ...
      && (~strcmp(Device, 'BIOSEMI') && ~strcmp(Device, 'BRAINAMP') ...
      && ~strcmp(Device, 'BASIC')))
	  [tmp_data] = vb_load_meg_data(filename{n});

  else
    
    loadspec.ChannelName = ch_names;
    loadspec.ActiveChannel = false;
    loadspec.ActiveTrial   = false;
    
    if n == 1
      % one channel_info is enough
  	  [tmp_data, channel_info] = vb_load_meg_data(filename{n}, loadspec);
    else
  	  [tmp_data] = vb_load_meg_data(filename{n}, loadspec);
    end
  end
  
  Nc = size(tmp_data,1);
  Nt = size(tmp_data,3);
  ix = (1:Nt) + Nst; % consecutive index for concatenated data
	
 	data(:,:,ix)  = tmp_data(:,:,:);
	
  Nst = Nt + Nst;
end

data = data(:,:,act_trial);
return

