function [fileinfo] = vb_get_multi_fileinfo(datafiles, outfile)
% Create Fileinfo-file which is to treat multiple MEG/EEG files as if 
% single MEG/EEG file. 
% [Usage]
%  [fileinfo] = vb_get_multi_fileinfo(datafiles, outfile);
% [Input]
%  datafiles : cell array of multiple data file names.
%              datafiles{n} : n-th file name  [n=1:Nsession]
%  outfile   : output file name to save multiple file information 'fileinfo'
%              if 'outfile' is not given, no output file is made
% [Output]
%  fileinfo  : structure with file information for multiple files
%  fileinfo.filename{n}  : n-th session data file names [n=1:Nsession]
%  fileinfo.Nchannel     : # of total channels
%  fileinfo.Nsample      : # of samples in one trial
%  fileinfo.Ntotal       : # of all trials 
%  fileinfo.Ntrial       : # of trials for each session [1 x Nsession]
%  fileinfo.session_id   : session index for each trial [1 x Ntotal]
%  fileinfo.cond_id      : condition number for each trials [1 x Ntotal]
%  fileinfo.ActiveChannel: Active channel list (boolean array) [Nchannel x 1]
%  fileinfo.ActiveTrial  : Active trial list   (boolean array) [Ntotal x 1]
%
% [History]
%   Masa-aki 2009-01-31
%   Sako     2009-09-01 substituted vb_load_measurement_info for vb_load_meg_info
%   Sako     2009-12-15 modified vb_load_measurement_info to return MEG format info
%   rhayashi 2017-04-20 unified Active information between MEG(EEGinfo).Active and fileinfo.
%                       (obsolete fields)channel_id, act_trial
%                       (new fields)     ActiveChannel, ActiveTrial
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

warning('Function ''vb_get_multi_fileinfo'' is obsolete and will be removed in a future release. Use ''vb_make_fileinfo'' instead.');

if ~iscell(datafiles)
	tmp = datafiles;
	datafiles = {tmp};
end;

% Number of EEG file
Nfile = length(datafiles);

%% ---- EEGinfo ---- %%
info     = vb_load_measurement_info(datafiles{1}, 'MEGINFO');
fsamp    = info.SampleFreq;
Nsample  = info.Nsample;
Nchannel = info.Nchannel;

% Trial number for each file
Ntrial    = zeros(1,Nfile);
trial_id  = cell(1,Nfile);
cond_id = [];

Nst = 0;

for n=1:Nfile
	info = vb_load_measurement_info(datafiles{n}, 'MEGINFO');
	
	if fsamp ~= info.SampleFreq,
		error('SampleFrequency does not match');
	end
	if Nsample ~= info.Nsample,
		error('NSample does not match');
	end

	if Nchannel ~= info.Nchannel,
		error('Nchannel does not match');
	end
	
	Ntrial(n) = info.Nrepeat;
	ix_trial = (1:Ntrial(n)) + Nst;
	Nst = Ntrial(n) + Nst;
	
	trial_id{n}  = ix_trial;
	
	if isfield(info,'cond_id')
		cond_id = [cond_id, info.cond_id];
	else
		cond_id = [cond_id, ones(1,Ntrial(n))];
	end
end

Ntotal = sum(Ntrial);
session_id = zeros(1,Ntotal);

for n=1:Nfile
	session_id(trial_id{n}) = n;
end

fileinfo.filename  = datafiles;
fileinfo.Ntotal    = Ntotal;
fileinfo.Ntrial    = Ntrial;
fileinfo.Nsample   = Nsample;
fileinfo.Nchannel  = Nchannel;
fileinfo.cond_id    = cond_id;
fileinfo.session_id = session_id;

% Active information
fileinfo.ActiveChannel = ones(Nchannel, 1);
fileinfo.ActiveTrial   = ones(Ntotal,   1);


%% ---- info ---- %%

if exist('outfile','var') && ~isempty(outfile)
	fprintf('Number of files: %d\n',Nfile)
	fprintf('Number of all trials: %d\n', Ntotal)
	fprintf('Save fileinfo: %s\n',outfile)
	
	Measurement = 'INFO';
	
	vb_fsave(outfile,'fileinfo','Measurement');
end

return
