function [Vout,F,xx,inf_C,xxA]= vb_load_cortex(brainfile, coord_type, analyzefile)
% load cortex vertex coordinate from brain file
%
% -- Syntax
% [Vout,F,xx,inf_C,xxA] = vb_load_cortex(brainfile,coord_type,analyzefile)
%
% -- Input
%   brainfile : <required>Brain file name
%             :
%  coord_type : <optional> <<string>> ID of coordinate.  case-insensitive.
%             : [not be specified]
%             : (default) return V of brainfile as 'Vout'
%             : 'SPM_Right_m' [m]
%             : 'Analyze_Right_voxel'  [voxel]
%             : 'Inflate'              
%             : 'Talairach': Talairach coordinate [mm]
%             : 'MNI'      : MNI-space coordinate [mm]
%             : 'subj'     : return subject cortex.
%             : If coord_type does not match with the all of above IDs,
%               it is interpreted as a user-specified ID and searching
%               struct variable 'Vall', if any. 
%             :
% analyzefile : <conditional required> corresponding to brainfile
%             : If the data is old format, or 'Analyze_Right_voxel' or 
%             : 'SPM_Right_m' is requested, analyzefile is needed.
%               
% --- Output
% Vout  : cortex vertex coordinate [Nvertex, 3]
% F     : Triangle patch index structure
% xx    : Normal vector to cortical surface [Nvertex, 3]
% inf_C : Color corresponding to Gyrus in inflate data [Nvertex, 1]
% xxA   : patch area assigned to each vertex [Nvertex, 1]
%
% written by M. Sato  2005-8-1
% Modified by Taku Yoshioka 2005-08-22
% 2007-03-12 (Sako) added two coord_types ('Talairach' and 'MNI')
% 2007-8-6 Masa-aki Sato bug fix for old format
% 2008-1-16 Masa-aki Sato 'V' is returned if there is no inflate data 
% 2010-11-18 Taku Yoshioka
%  User-specified coordinate type supported. 
%  Message display changed (vb_disp). 
% 2017-1-25 rhayashi Added coord_type 'subj'
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

inf_C = [];

% --- check brain file --- %
if ~exist('brainfile', 'var'), brainfile = []; end;
[brain_id] = inner_check_brainfile(brainfile);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- DEFAULT
%
if ~exist('coord_type','var'),
  load(brainfile,'V','F','xx','xxA');
  Vout = V;
  return;
end;

load(brainfile,'V*','F','xx','xxA');

switch lower(coord_type)
  % --- Inflate
  %
 case  'inflate'
  if exist('Vinflate','var')
    Vout = Vinflate;
    load(brainfile,'inf_C'); 
  else
    inflated_brainfile = [brain_id '.inflate.mat'];
    if exist(inflated_brainfile, 'file')
      load(inflated_brainfile, 'V','inf_C')
      Vout = V;
    else
      Vout = V;
      inf_C = ones(size(V,1),1);
      vb_disp(['There is no inflate brain data:' brainfile ]);
    end;
  end;

  % --- SPM_Right_m
  %
 case  'spm_right_m'
  if exist('Vinfo','var')
    Vout = V;
  else
    vb_disp('Old SBI brain file ');
    [Vdim, Vsize] = analyze_hdr_read(analyzefile);
    Vout = vb_vox_to_spm_right(Vox,Vdim,Vsize)*0.001;% mm -> m
  end;
    
  % --- Analyze_Right_voxel
  %
 case  'analyze_right_voxel'
  if exist('Vinfo','var')
    if exist('Vox','var')
      Vout = Vox;
    else
      [Vdim, Vsize] = analyze_hdr_read(analyzefile);
      Vout = vb_spm_right_to_analyze_right(V,Vdim,Vsize);
    end;
  else
    vb_disp('Old SBI brain file ');
    [Vdim, Vsize] = analyze_hdr_read(analyzefile);
    Vout = vb_vox_to_analyze_right(Vox,Vdim,Vsize);
  end;
    
  % --- Talairach
  %
 case  'talairach'
  if exist('Vtal', 'var')
    Vout = Vtal;
  else
    xyzfile = [brain_id  '.xyz.mat'];
    if exist(xyzfile, 'file')
      load(xyzfile, 'XYZtal');
      Vout = XYZtal;
    else
      Vout = [];
    end;
  end;
    
  % --- MNI
  %
 case  'mni'
  if exist('Vmni', 'var')
    Vout = Vmni;
  else
    xyzfile = [brain_id  '.xyz.mat'];
    if exist(xyzfile, 'file')
      load(xyzfile, 'XYZmm');
      Vout = XYZmm;
    else
      Vout = [];
    end;
  end;
  
  if isfield(F,'Fmni')
  	F = F.Fmni;
  end
 case  'subj'
     warning('off', 'MATLAB:load:variableNotFound');
     load(brainfile, 'subj');
     warning('on',  'MATLAB:load:variableNotFound');
     if exist('subj', 'var')
         % for V2.x format
         Vout  = subj.V;
         F     = subj.F;
         xx    = subj.xx;
         inf_C = [];
         if isfield(subj, 'xxA')
             % v2.1 format
             xxA = subj.xxA;
         else
             % v2.0 format
             % xxA is standard brain's one.
         end
     else
         % for V1 format
         [Vout,F,xx,inf_C,xxA]= vb_load_cortex(brainfile);
     end
 otherwise
  if ~exist('Vall'), 
    error('Unknown coord_type : %s', coord_type);
  end;
  
  Vout = [];
  for i=1:length(Vall.key)
    if strcmp(Vall.key{i},coord_type), 
      Vout = Vall.V{i};
      break;
    end;
  end;
  
  if isempty(Vout), 
    error('Unknown coord_type : %s', coord_type);
  end;
end;

return;

%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_brainfile()
%
function [brain_id] = inner_check_brainfile(brainfile)

if isempty(brainfile), error('brainfile is a required parameter'); end;

if ~isempty( findstr(brainfile, '.brain.mat') )
	brain_id = brainfile(1:findstr(brainfile, '.brain.mat')-1);
else
	error(['Brain file name is not correct:' brainfile]);
end

if exist(brainfile, 'file') ~= 2
  error('cannot read brainfile : %s', brainfile);
end
%
% --- end of inner_check_brainfile()

%%% END OF FILE %%%
