function	[meginfo] = vb_load_meg_info(megfile, info_type)
% return MEGinfo type data from MEG-MAT file or EEG-MAT file
%
% [usage]
%    [meginfo] = vb_load_meg_info(megfile, info_type);
%
% [input]
%     megfile : <required> <<file>> measurement file
%             :  meg-mat file(.meg.mat) or eeg-mat file(.eeg.mat)
%
%   info_type : <optional> (bit sequence) type code you want [1]
%             :  1) (0001) basic
%             :      .SampleFreq
%             :      .Nchannel
%             :      .Nsample
%             :      .Nrepeat
%             :      .Pretrigger
%             :      .device
%             :      .sensor_weight (MEG)
%             :      .Coord (EEG)
%             :      .Measurement
%             :  2) (0010) for brain sphere model
%             :      .Vcenter
%             :      .Vradius
%             :  4) (0100) channel data
%             :      .MEGch_id
%             :      .MEGch_name
%             :      .ChannelInfo
%             :      .ExtraChannelInfo
%             :      .saveman
%             :  8) (1000) active conditions
%             :      .ActiveChannel
%             :      .ActiveTrial
%             :      .Trial
%             :  [NOTICE]
%             :  You have to set *SUMMATION* value of bits by *DECIMAL* number.
%             :  (combination example)
%             :   3 (0011) --> basic + brain sphere model
%             :   5 (0101) --> basic + channel data
%             :   6 (0110) --> brain sphere model + channel data
%             :   7 (0111) --> basic + brain sphere model + channel data
%             :   9 (1001) --> basic + active conditions
%             :  13 (1101) --> basic + channel_data + active conditions
%             :  15 (1111) --> all
%
% [output]
%    meginfo : newest format - sbi      @ see make_sbi_meg_data()
%                            - yokogawa @ see make_yokogawa_data()
%
% [note]
%   standard format:
%   meginfo
%    .Measurement   % measurement type
%    .device        % meg device type
%    .Nchannel      % number of active gradiometers
%    .Nsample       % sampling number of measurement
%    .Nrepeat       % repeat number
%    .Pretrigger    % sampling number for pre-trigger
%    .SampleFreq    % sampling frequency
%    .Vcenter       % origin of sphere model [m]
%    .Vradius       % radius of sphere model [m]
%    .ActiveChannel % [Nchannel x 1 boolean] list of active channel flag
%    .ActiveTrial   % [Ntrial x 1 boolean] list of active trial flag
%    .MEGch_id
%    .MEGch_name
%    .MEG_ID
%    .MRI_ID
%    .Trial
%    .ChannelInfo
%    .ExtraChannelInfo
%    .saveman
%
% measurement-dependent members are:
%    .sensor_weight % for MEG
%    .Coord         % for EEG
%
%   if given is an eeg-mat file, convert EEGinfo to MEGinfo
%
%   See also
%     vb_load_measurement_info
%
% [history]
%   2006-12-13 (Sako) modified for EEG
%   2008-03-10 (Sako) renamed fields
%   2008-04-15 (Sako) added Trial, Active*
%   2009-08-03 (Sako) added info_type and changed the set of fields
%   2010-01-22 (Sako) modified bug of INFOTYPE_ACTIVE
%   2010-09-16 (Sako) modified to inherit info_type when 'INFO' type file
%   2011-03-08 (Sako) added 'sensor_weight' for MEG and 'Coord' for EEG
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%   2011-06-02 (Sako) modified according to the new data format
%   2012-07-13 (Sako) supported an MEG-MAT file which has 'Pretriger' field.
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('megfile', 'var'), megfile = ''; end
if ~exist('info_type', 'var'), info_type = []; end
[megfile, info_type] = inner_check_arguments(megfile, info_type);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
[measurement] = vb_load_device(megfile);
Measurement = upper(measurement);

switch  Measurement

  case  'MEG'
    % this script is in original vb_load_meg_info
    load(megfile, 'MEGinfo');

  case  'EEG'
    MEGinfo = vb_load_measurement_info(megfile, 'meginfo');

  case  'INFO'
    load(megfile, 'fileinfo');
    fileinfo = vb_fileinfo_active_field_convert_to('new', fileinfo, megfile);

    MEGinfo = vb_load_meg_info(fileinfo.filename{1}, info_type);
    MEGinfo.ActiveChannel = fileinfo.ActiveChannel;
    MEGinfo.ActiveTrial   = fileinfo.ActiveTrial;
    
    MEGinfo.Nrepeat       = fileinfo.Ntotal;
  otherwise
    error('unknown Measurement : %s', Measurement);
end

meginfo = inner_read_meg_info(MEGinfo, Measurement, info_type, megfile);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [megfile, info_type] = inner_check_arguments(megfile, info_type)
func_ = mfilename;

if isempty(megfile)
  error('(%s) megfile is a required parameter', func_);
end

if isempty(info_type)
  info_type = 1; % basic
end
return;
%
% --- end of inner_check_arguments()


% --- inner_read_meg_info()
%
function [meginfo] = ...
  inner_read_meg_info(MEGinfo, Measurement, info_type, megfile)
func_ = mfilename;

meginfo = [];

% --- constants
INFOTYPE_BASIC   = 1; % 0001
INFOTYPE_SPHERE  = 2; % 0010
INFOTYPE_CHANNEL = 4; % 0100
INFOTYPE_ACTIVE  = 8; % 1000

% --- BASIC TYPE
if bitand(info_type, INFOTYPE_BASIC) == INFOTYPE_BASIC
  % --- .SampleFreq
  if isfield(MEGinfo, 'SampleFreq')
    meginfo.SampleFreq = MEGinfo.SampleFreq;
  else
    warning('(%s) megfile(%s) does not have SampleFreq\n', func_, megfile);
  end

  % --- .Nchannel
  if isfield(MEGinfo, 'Nchannel')
    meginfo.Nchannel = MEGinfo.Nchannel;
  else
    warning('(%s) megfile(%s) does not have Nchannel\n', func_, megfile);
  end

  % --- .Nsample
  if isfield(MEGinfo, 'Nsample')
    meginfo.Nsample = MEGinfo.Nsample;
  else
    warning('(%s) megfile(%s) does not have Nsample\n', func_, megfile);
  end

  % --- .Nrepeat
  if isfield(MEGinfo, 'Nrepeat')
    meginfo.Nrepeat = MEGinfo.Nrepeat;
  else
    warning('(%s) megfile(%s) does not have Nrepeat\n', func_, megfile);
  end

  % --- .Pretrigger
  meginfo.Pretrigger = vb_meginfo_get_pre_trigger(MEGinfo);

  % --- .Measurement
  if isfield(MEGinfo, 'Measurement')
    meginfo.Measurement = MEGinfo.Measurement;
  else
%     warning('(%s) megfile(%s) does not have Measurement\n', func_, megfile);
  end

  % --- .device
  if isfield(MEGinfo, 'device')
    meginfo.device = MEGinfo.device;
  else
    warning('(%s) megfile(%s) does not have device\n', func_, megfile);
  end
  
  % --- (MEG) .sensor_weight
  if strcmp(Measurement, 'MEG') == 1
    if isfield(MEGinfo, 'sensor_weight')
      meginfo.sensor_weight = MEGinfo.sensor_weight;
    else
      warning('(%s) megfile(%s) does not have sensor_weight\n', func_, megfile);
    end
  end
  
  % --- (EEG) .Coord
  if strcmp(Measurement, 'EEG') == 1
    EEGinfo = vb_eegfile_load_eeginfo(megfile);
    if isfield(EEGinfo, 'Coord')
      meginfo.Coord = EEGinfo.Coord;
    else
      warning('(%s) megfile(%s) does not have Coord\n', func_, megfile);
    end
  end
end
    
if bitand(info_type, INFOTYPE_SPHERE) == INFOTYPE_SPHERE
  % --- .Vcenter
  if isfield(MEGinfo, 'Vcenter')
    meginfo.Vcenter = MEGinfo.Vcenter;
  else
    warning('(%s) megfile(%s) does not have Vcenter\n', func_, megfile);
  end

  % --- .Vradius
  if isfield(MEGinfo, 'Vradius')
    meginfo.Vradius = MEGinfo.Vradius;
  else
    warning('(%s) megfile(%s) does not have Vradius\n', func_, megfile);
  end
end
    
if bitand(info_type, INFOTYPE_CHANNEL) == INFOTYPE_CHANNEL
  % --- .MEGch_id
  if isfield(MEGinfo, 'MEGch_id')
    meginfo.MEGch_id = MEGinfo.MEGch_id;
  else
    warning('(%s) megfile(%s) does not have MEGch_id\n', func_, megfile);
  end

  % --- .MEGch_name
  if isfield(MEGinfo, 'MEGch_name')
    meginfo.MEGch_name = MEGinfo.MEGch_name;
  else
    warning('(%s) megfile(%s) does not have MEGch_name\n', func_, megfile);
  end

  % --- .ChannelInfo
  if isfield(MEGinfo, 'ChannelInfo')
    meginfo.ChannelInfo = MEGinfo.ChannelInfo;
  else
    warning('(%s) megfile(%s) does not have ChannelInfo\n', func_, megfile);
  end

  % --- .ExtraChannelInfo
  if isfield(MEGinfo, 'ExtraChannelInfo')
    meginfo.ExtraChannelInfo = MEGinfo.ExtraChannelInfo;
  else
    warning('(%s) megfile(%s) does not have ExtraChannelInfo\n', ...
      func_, megfile);
  end

  % --- .saveman
  if isfield(MEGinfo, 'saveman')
    meginfo.saveman = MEGinfo.saveman;
  else
    warning('(%s) megfile(%s) does not have saveman\n', func_, megfile);
  end
end
    
if bitand(info_type, INFOTYPE_ACTIVE) == INFOTYPE_ACTIVE
  % --- .ActiveChannel
  if isfield(MEGinfo, 'ActiveChannel')
    meginfo.ActiveChannel = MEGinfo.ActiveChannel;
  else
    warning('(%s) megfile(%s) does not have ActiveChannel\n', func_, megfile);
  end

  % --- .ActiveTrial
  if isfield(MEGinfo, 'ActiveTrial')
    meginfo.ActiveTrial = MEGinfo.ActiveTrial;
  else
    warning('(%s) megfile(%s) does not have ActiveTrial\n', func_, megfile);
  end

  % --- .Trial
  if isfield(MEGinfo, 'Trial')
    meginfo.Trial = MEGinfo.Trial;
  else
    warning('(%s) megfile(%s) does not have Trial\n', func_, megfile);
  end
end
return;
%
% --- enf of inner_read_meg_info()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

%%% END OF FILE %%%
