function [sensor, center, result, channel_info] = vb_load_eeg_sensor(eegfile, active_swt)
% return coordinates of sensors from EEG-MAT file
% [usage]
%   [sensor, center, result] = vb_load_eeg_sensor(eegfile, active_swt)
% [input]
%      eegfile : <required> <<file>> EEG-MAT file
%   active_swt : <optional> <<boolean>> [false] switch to filter channel
%              :    true) return only active channels (MEG)
%              :   false) return all the channels (MEG)
%              : only when mode is 'MEG', this switch is available
% [output]
%       sensor : coordinates of sensors which are loaded from eegfile
%              :  [Nchannel x 3]
%       center : coordinate of center point
%              :  [1 x 3]
%       result : <<integer>>
%              :  0) success
%              :  1) error - bad eegfile
%              :  2) error - not be EEG data
% channel_info : <<struct>> channel information of loaded data
%                :  .Active [Nchannel x 1]
%                :  .Name   [Nchannel x 1]
%                :  .Type   [Nchannel x 1]
%                :  .ID     [Nchannel x 1]
% [note]
%
% [history]
%   2006-12-11 (Sako) initial version
%   2008-04-16 (Sako) added active_swt and rewrote
%   2010-10-22 (Sako) modified help - output > sensor
%   2011-06-01 (Sako) converted return values of vb_load_device to upper case
%   2017-03-06 (rhayashi) Add channel_info
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('eegfile', 'var'), eegfile = ''; end
if ~exist('active_swt', 'var'), active_swt = []; end
[eegfile, active_swt] = inner_check_arguments(eegfile, active_swt);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%
func_ = mfilename;

sensor = [];
center = [];
channel_info = [];

if isempty(eegfile)
  fprintf('(%s)eegfile (%s) is invalid\n', func_, eegfile);
  result = 1;
  return;
end

[measurement] = vb_load_device(eegfile);
Measurement = upper(measurement);

if ~strcmp(Measurement, 'EEG')
  fprintf('(%s) ERROR: %s is not eeg data file\n', func_, eegfile);
  result = 2;
  return;
end

EEGinfo = vb_load_measurement_info(eegfile);
[sensor, center, result, channel_info] = vb_eeginfo_get_sensor_position(EEGinfo, active_swt);
return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [eegfile, active_swt] = inner_check_arguments(eegfile, active_swt)
if isempty(eegfile)
  eegfile = '';
end

if isempty(active_swt)
  active_swt = false;
end
return;
%
% --- end of inner_check_arguments()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %
