function [Vinflate,inf_C] = vb_make_inflate_data(proj_root, brain_parm)
% Incorporate inflated model of Brain Voyager/FreeSurfer. 
% These data are associated with the cortical surface model from which 
% inflate or flat map are created. This function needs .brain.mat file,
% created based on the brain surface data, to make correspondence 
% of vertices between the original brain and inflated model. 
% Variables 'BV_index'  in .brain.mat file are used to make
% correspondence, i.e., reduction of vertices from the brain
% surface data to VBMEG cortical surface model (.brain.mat). 
%
% [syntax]
% [Vinflate,inf_C] = vb_make_inflate_data(proj_coor,brain_parm)
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% brain_parm: <<structure>> Parameters for creating inflated model
% --- fields of brain_parm
%  brain_file  : <<string>> Filename of .brain.mat file
%
%  (parameters for BrainVoyager inflated model)
%  BV_left_infl_file : <<string>> Brain-Voyager left  inflate file
%  BV_right_infl_file: <<string>> Brain-Voyager right inflate file
%
%  (parameters for FreeSurfer inflated model)
%  FS_left_infl_file : <<string>> FreeSurfer left inflate file
%  FS_right_infl_file: <<string>> FreeSurfer right inflate file
%  FS_left_curv_file : <<string>> FreeSurfer left curvature file
%  FS_right_curv_file: <<string>> FreeSurfer right curvature file
% ---
%
% [output]
% Vinflate: <<matrix>> Inflate model vertex
% inf_C   : <<vector>> Color corresponding to curvature
%
% [history]
% Ver 1.0 written by M. Sato  2003-3-15
% Ver 2.0  modified by M. Sato  2005-4-8
%  SPM right-handed coordinate system is emplyoed. 
% Ver 2.1  modified by M. Sato  2005-8-9
%  Drastic revision of variables.
% 2010-05-28 Taku Yoshioka
%  Minor change (verbose level support)
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

%
% --- Brain-Voyager coordinate
%
% [Left-hand coordinate]
% X:Front(0) -> Back(255)
% Y:Top(0)   -> Bottom(255)
% Z:Right(0) -> Left(255)

% V0L, V0R, LColor, RColor 
tic; 
if isfield(brain_parm,'BV_left_infl_file') & ...
      ~isempty(brain_parm.BV_left_infl_file),
  vb_disp_nonl('Make inflate model using BrainVoyager SRF files: ');
  filetype = 'BrainVoyager'; 
  
  Srf = vb_ScriptLoadSRF(brain_parm.BV_left_infl_file);

  V0L    = [Srf.VertexX, Srf.VertexY, Srf.VertexZ]; 
  LColor = Srf.MeshColor;

  Srf    = vb_ScriptLoadSRF(brain_parm.BV_right_infl_file);

  V0R    = [Srf.VertexX, Srf.VertexY, Srf.VertexZ]; 
  RColor = Srf.MeshColor;

  clear Srf
elseif isfield(brain_parm,'FS_left_infl_file') & ...
      ~isempty(brain_parm.FS_left_infl_file), 
  vb_disp_nonl('Make inflate model using FreeSurfer .asc files: ');
  filetype = 'FreeSurfer'; 
  
  [V0L,F0L] = vb_fs_load_surface(brain_parm.FS_left_infl_file);
  [V0R,F0R] = vb_fs_load_surface(brain_parm.FS_right_infl_file);
  
  clear F0L F0R
  
  [N0L,LColor] = vb_fs_load_normal(brain_parm.FS_left_curv_file);
  [N0R,RColor] = vb_fs_load_normal(brain_parm.FS_right_curv_file);

  xmean = mean([V0L(:,1); V0R(:,1)]);
  xwidth = max(abs([V0L(:,1); V0R(:,1)])-xmean);
  V0L(:,1) = V0L(:,1)-xwidth; 
  V0R(:,1) = V0R(:,1)+xwidth; 
  
  LColor = (sign(LColor)+1)/2; 
  RColor = (sign(RColor)+1)/2; 
  
  clear N0L N0R
end
vb_disp(sprintf('%f[sec]',toc));

%
% --- redeuce patchs
%
if isempty(proj_root)
	brain_file = [brain_parm.brain_file];
else
	brain_file = [proj_root filesep brain_parm.brain_file];
end
load([brain_file],'BV_index','Vinfo');

V2L = V0L(BV_index.Left ,:);
V2R = V0R(BV_index.Right-Vinfo.NdipoleL0,:);

inf_LC	= LColor(BV_index.Left ,:);
inf_RC	= RColor(BV_index.Right-Vinfo.NdipoleL0,:);

switch filetype, 
 case 'BrainVoyager', 
  %
  % --- Analyze voxcel coordinate   
  %
  % [Right-hand coordinate]
  % X: Left   -> Right
  % Y: Back   -> Front
  % Z: Bottom -> Top

  V2L = vb_bvoyger_to_vox(V2L);
  V2R = vb_bvoyger_to_vox(V2R);

  inf_V3L = vb_vox_to_analyze_right(V2L);
  inf_V3R = vb_vox_to_analyze_right(V2R);
  
 case 'FreeSurfer', 
  inf_V3L = V2L; 
  inf_V3R = V2R; 
end; 

% Total vertex
% inf_V3 -> V
Vinflate = [inf_V3L ; inf_V3R];   
inf_C	 = [inf_LC  ; inf_RC ];   
