function [V,F,xx,BV_index,Vinfo] = vb_make_brain_std(brain_parm)
% Make cortex data for MNI standard brain
% 
%   [V,F,xx,ORG_index,Vinfo] = vb_make_brain_std(brain_parm)
%
%--- Input
%
% brain_parm   : structure with following field
% --- fields of brain_parm
% .FS_left_file         = [fs_dir '/bem/lh.smoothwm.asc'];
% .FS_right_file        = [fs_dir '/bem/rh.smoothwm.asc'];
% 
% .FS_left_label_file  = [fs_dir '/label/lh.cortex.label'];
% .FS_right_label_file = [fs_dir '/label/rh.cortex.label'];
%
%  .FS_left_sphere_file
%  .FS_right_sphere_file
%  .FS_sphere_key 
%
%    .Nvertex        : Total number of vertices of the right and left brain
%                      after reducing original(BV/FS) model.
%                      (Priority: Nvertex>reduce_ratio)
%    .N_step         : Division number in searching match points
%         ĺбõZʬ
% 
%--- Output
% 
% V    : Cortical vertex point cordinate (SPM_Right_m) [Nvertex, 3]
%          ĺ  ĺο, 3
% xx   : Normal vector to cortical surface   [Nvertex, 3]
%           ĺˡȰ֡ ĺο, 3
% F      : Patch index structure
%           (3ѷ)룳Ĥĺֹ ( ̤ο, 3)  
%  .F3R	 : Right cortex
%  .F3L	 : Left  cortex
%  .F3	 : Left + Right
% BV_index  : original vertex index in BV corresponding to brain
%    .Left     : Left  brain
%    .Right    : Right brain
% Cortex area index of original FreeSurfer surface
% BV_index.cortexL 
% BV_index.cortexR 
% Vinfo        : Vertex dimension structure
%   .Ndipole   : # of vertex
%   .NdipoleL  : # of vertex in Left cortex
%   .Npatch    : # of patch
%   .Coord     = 'SPM_Right_m';
% Cortex area index of 'V'
% Vinfo.cortexL = 
% Vinfo.cortexR = 
%
% [history]
% Ver.2.0 New version for group analysis
% 2015-12-27 M. Sato 
%
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

plot_mode = 1;

global vbmeg_inst;
const = vb_define_verbose;

% Max radius to search corresponding point [mm] in the 1st-step
Rmax = 4; 
EPS  = 1e-10;

% --- Subject cortical surface (SPM_Right mm cordinate)
vb_disp('Load original brain surface');

[V0L,F0L,n0L,V0R,F0R,n0R] = vb_load_orig_brain_surf(brain_parm);
% V0L/R  vertex coordinate (SPM_Right mm)
% n0L/R  unit vector 
% FOL/R  triangle patch

% Extract cortex area index
vb_disp('Extract cortex area index');

[Vindex] = vb_extract_cortex(V0L,F0L,V0R,F0R,brain_parm,1);

vb_disp('Reduce cortex');

% reference coordinate: subject original SPM-R mm coordinate
VLref = V0L;
VRref = V0R;

R = 1.0;
if isfield(brain_parm, 'Nvertex') && ~isempty(brain_parm.Nvertex) &&...
   isnumeric(brain_parm.Nvertex)
   % Number of verticies after reducing
   N = brain_parm.Nvertex;
   if((size(V0L, 1) + size(V0R, 1)) < N)
     vb_disp(['brain_parm.Nvertex is larger than original model''s one.'...
              'No reduction is applied.'], const.VERBOSE_LEVEL_WARNING);
   else
       R = N;
       vb_disp(sprintf('Nvertex = %d is applied.', brain_parm.Nvertex));
   end
elseif isfield(brain_parm, 'reduce_ratio') && ...
       isnumeric(brain_parm.reduce_ratio)
    R = brain_parm.reduce_ratio;
    vb_disp(sprintf('reduce_ratio = %.2f is applied.', ...
            brain_parm.reduce_ratio));
end

% target coordinate: 
% Subject reduced SPM-R mm cordinate
[FL,VL] = vb_reducepatch( F0L, V0L, R ); % 0 < R < 1: ratio
[FR,VR] = vb_reducepatch( F0R, V0R, R ); % R > 1   : number of verticies

[FL, VL, xxL] = vb_out_normal_surf(FL,VL);
[FR, VR, xxR] = vb_out_normal_surf(FR,VR);

%
% ---- Check closed surface by solid_angle/(2*pi) = 1
%
omega  = vb_solid_angle_check(VL,FL);
vb_disp(sprintf('Left solid_angle/(2*pi) = %5.3f',omega));
if abs(omega - 1) > EPS,
	vb_disp('Surface is not closed ');
end
omega  = vb_solid_angle_check(VR,FR);
vb_disp(sprintf('Right solid_angle/(2*pi) = %5.3f',omega));
if abs(omega - 1) > EPS,
	vb_disp('Surface is not closed ');
end

NdipoleL = size(VL,1);
NdipoleR = size(VR,1);
Ndipole  = NdipoleL + NdipoleR;
FR = FR + NdipoleL;

F.F3	   = [FL ; FR];
F.F3R	   = FR;
F.F3L	   = FL;
F.NdipoleL = NdipoleL;

% Find nearest point from 'Vref' for 'V'
%  [Indx ,ddmin] = vb_find_nearest_point(Vref, V, Rmax,Nstep,Mode,Disp)
% Vref  : Ref vertex point 
% Indx  : Ref vertex index nearest to V

% V0L/R : original vertex coordinate (SPM_Right mm)
vb_disp('Find original point corresponding to reduced model');

tic;
vb_disp_nonl('Left cortex: ');
[IndxL, ddL] = vb_find_nearest_point(VLref, VL, Rmax, 100, 1, 1);
vb_disp(sprintf('%f[sec]',toc));

tic;
vb_disp_nonl('Right cortex: ');
[IndxR, ddR] = vb_find_nearest_point(VRref, VR, Rmax, 100, 1, 1);

vb_disp(sprintf('%f[sec]',toc));

dd = [ddL; ddR];
dmax = max(dd);
fprintf('Max distance = %4.1f mm \n', dmax)

%
% --- SPM cordinate in [m] , Normal vector is outward.
%
V  = [VL; VR]/1000;

% ---  Normal vector of reduced surface
xx  = [xxL ; xxR]; 

% Dimension of Original vertex 
NdipoleL0 = size(VLref,1);
NdipoleR0 = size(VRref,1);
Npatch    = size(F.F3,1);

% Original vertex index corresponding to reduced brain 'V'
BV_index.Left  = IndxL; 
BV_index.Right = IndxR + NdipoleL0;

% Cortex area index of original FreeSurfer surface
BV_index.cortexL = Vindex.cortexL;
BV_index.cortexR = Vindex.cortexR + NdipoleL0;

% Dimensional info
Vinfo.Ndipole   = Ndipole;
Vinfo.NdipoleL  = NdipoleL;
Vinfo.NdipoleR  = NdipoleR;

Vinfo.Ndipole0  = NdipoleL0 + NdipoleR0;
Vinfo.NdipoleL0 = NdipoleL0;
Vinfo.NdipoleR0 = NdipoleR0;

Vinfo.Npatch    = Npatch;
Vinfo.Coord     = 'SPM_Right_m';
Vinfo.dd_match  = dd;

% Cortex area index of reduced surface
flgL = zeros(NdipoleL0,1);
flgR = zeros(NdipoleR0,1);

% Cortex flag
flgL(Vindex.cortexL) = 1;
flgR(Vindex.cortexR) = 1;

% Reduced index table
tblL = zeros(NdipoleL0,1);
tblR = zeros(NdipoleR0,1);

tblL(IndxL) = 1:NdipoleL;
tblR(IndxR) = 1:NdipoleR;

% Get cortex index in Reduced surface
ixL  = find( tblL .* flgL > 0);
ixR  = find( tblR .* flgR > 0);

Vinfo.cortexL = tblL(ixL);
Vinfo.cortexR = tblR(ixR) + NdipoleL;

if plot_mode == 0, return; end;

% plot
nonL = vb_setdiff2([1:NdipoleL], Vinfo.cortexL);
nonR = vb_setdiff2([(NdipoleL+1):Ndipole], Vinfo.cortexR);

FLz = vb_patch_select(nonL, F.F3L, Ndipole);
FRz = vb_patch_select(nonR, F.F3R, Ndipole);

figure;
subplot(1,2,1)
vb_plot_surf(V,F.F3L,'','b')
hold on
vb_plot_surf(V,FLz,'r','r')

subplot(1,2,2)
vb_plot_surf(V,F.F3R,'','b')
hold on
vb_plot_surf(V,FRz,'r','r')

