function [V,F,xx,BV_index,Vinfo] = ...
	make_std_brain_model(proj_root,brain_parm, DEBUG_MODE)
% Make standard brain from ICBM152
% (VBMEG public function)
%
% [syntax]
% make_std_brain_model(proj_root,brain_parm);
%
% [input]
% proj_root : <<string>> VBMEG project root directory. 
% brain_parm: <<struct>> Parameters for creating cortical surface model
% --- fields of brain_parm
%  analyze_file: <<string>> MRI filename (.hdr)
%
% brain_parm.FS_left_file         = [fs_dir '/bem/lh.smoothwm.asc'];
% brain_parm.FS_right_file        = [fs_dir '/bem/rh.smoothwm.asc'];
% brain_parm.FS_left_infl_file    = [fs_dir '/bem/lh.inflated.asc'];
% brain_parm.FS_right_infl_file   = [fs_dir '/bem/rh.inflated.asc'];
% brain_parm.FS_left_curv_file    = [fs_dir '/bem/lh.curv.asc'];
% brain_parm.FS_right_curv_file   = [fs_dir '/bem/rh.curv.asc'];
% brain_parm.FS_left_label_file  = [fs_dir '/label/lh.cortex.label'];
% brain_parm.FS_right_label_file = [fs_dir '/label/rh.cortex.label'];
% 
% 	brain_parm.FS_left_sphere_file  = [fs_dir '/bem/lh.sphere.reg.asc'];
% 	brain_parm.FS_right_sphere_file = [fs_dir '/bem/rh.sphere.reg.asc'];
% 	
% 	brain_parm.FS_sphere_key = 'sphere.reg';
% 
%  brain_file  : <<string>> Cortical surface model file (.brain.mat).
%  act_file    : <<string>> Cortical activity map file (.act.mat).
%  area_file   : <<string>> Cortical area file (.area.mat).
%  Nvertex     : <<int>>    Total number of vertices for cortical surface model
%  Rmax        : <<float>>  Maximum radius for neighbor search 
%  N_step      : <<int>>    Division number of z axis in matching between
%                           cortical surface model and MRI points
%  display     : <<int>>    Display progress for this steps
%  
% ---
% 
% [output]
% The following files are created: 
%  VBMEG cortical surface model file (.brain.mat)
%  VBMEG area file (.area.mat)
%  VBMEG activity map file (.act.mat)
%
% The following variables are stored in .brain.mat file: 
% V         : Cortical vertex point cordinate (SPM_Right_m) 
% F         : Patch index structure
% xx        : Normal vector to cortical surface 
% xxA       : area asigned for each vertex ([m^2])
%
% BV_index  : original vertex index in BV/FS corresponding to brain
% Vinfo     : Vertex dimension structure
%
% xxF{i}    : Next-neighbor index for the vertex-i
% xxD{i}    : Distance between next-neighbor and the vertex-i
% xxT{n}    : vertex index for n-th triangle
% nextIX{i} : Neighbor index list for the vertex-i
% nextDD{i} : Distance from the vertex-i
% 
% The following areas are automatically created and stored in .area.mat
% file: 
% areakey = 'Cortex'         : whole brain
% areakey = 'cortex_region'  : cortex region without corpus area
% areakey = 'corpus_area'    : corpus area
%
% [history]
% Ver.2.0 New version for group analysis
% 2015-12-27 M. Sato 
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

global  vbmeg_inst
const = vb_define_verbose;
VERBOSE_LEVEL_NOTICE = const.VERBOSE_LEVEL_NOTICE;

% Set BV-SPM version (mode = 4 : std_brain mode)
if ~exist('DEBUG_MODE','var'), DEBUG_MODE = 0; end;

% Define constant
define = vbmeg_inst.const;

proj_root = vb_rm_trailing_slash(proj_root);
if exist(proj_root, 'dir') ~= 7
    vb_mkdir(proj_root);
end

%
% Do not modify following lines
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

% Output file path
if isempty(proj_root)
	brain_file = [brain_parm.brain_file];
	area_file  = [brain_parm.area_file ];
	act_file   = [brain_parm.act_file  ];
else
	brain_file = [proj_root filesep brain_parm.brain_file];
	area_file  = [proj_root filesep brain_parm.area_file ];
	act_file   = [proj_root filesep brain_parm.act_file  ];
end

% ID for MRI image data
analyze_file = brain_parm.analyze_file;

[udir, fname] = fileparts(analyze_file);
MRI_ID = fname;

flag_fs = (isfield(brain_parm, 'FS_left_sphere_file') ... 
       &&    ~isempty(brain_parm.FS_left_sphere_file) ...
       &&  isfield(brain_parm, 'FS_right_sphere_file') ...
       &&   ~isempty(brain_parm.FS_right_sphere_file) ...
       &&  isfield(brain_parm, 'FS_sphere_key') ...
       &&   ~isempty(brain_parm.FS_sphere_key));

if flag_fs==0,
	error('FreeSurfer file setting error');
end

Rmax    = brain_parm.R_max; 
Display = brain_parm.display;

%-------------------------------------------------------
% Make cortex coordinate from FreeSurfer
%-------------------------------------------------------
[V,F,xx,BV_index,Vinfo] = vb_make_brain_std(brain_parm);

if DEBUG_MODE==2, return; end;

%--------------------------------------------------------------------------
% Calculate MNI coordinate and Talairach coordinate
%--------------------------------------------------------------------------

Vmni = convert_spm_mni(V, analyze_file)*0.001;
Vtal = mni2tal(Vmni);

%
% Save cortical surface model
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp(['Save brain model '], VERBOSE_LEVEL_NOTICE);
vb_disp(sprintf('     filename = %s', brain_file), ...
        VERBOSE_LEVEL_NOTICE);
vb_save([brain_file],...
     'F','V','xx','Vinfo','BV_index','MRI_ID','Vmni','Vtal');

%----------------------------------------
% Search next-point index and distance
%----------------------------------------
tic;
vb_disp_nonl('Search next-point index and distance ', ...
        VERBOSE_LEVEL_NOTICE);

[xxD, xxF, xxT] = vb_next_distance( F.F3, V ); % unit : m

vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

% total number of vertex
Ndipole = Vinfo.Ndipole;
Vindx   = 1:Ndipole;

%----------------------------------------------
% Calculate area assigned to the vertex
%----------------------------------------------
tic; 
[xxA] = vb_calc_patch_area(V, F.F3, xxT);
vb_disp(sprintf('%f[sec]',toc), VERBOSE_LEVEL_NOTICE);

%
% Save next neighbor index
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
vb_disp(['Save next neighbor index'], VERBOSE_LEVEL_NOTICE);
vb_save([brain_file], 'xxF','xxD','xxA');

%---------------------------------------------------
% Find original vertex near the cortical vertex
%---------------------------------------------------
normal_stat = vb_original_normal_statics(proj_root,brain_parm,BV_index);

% normal average over neighbor points of original brain
xx = vb_calc_normal_average(normal_stat.neighbor_org, normal_stat.normal_org);

vb_disp('Save neighbor statics in the original brain', ...
        VERBOSE_LEVEL_NOTICE);
vb_save(brain_file,'normal_stat','xx');

%--------------------------------------------------------------------------
% Save FreeSurfer sphere coordinate
%--------------------------------------------------------------------------
	parm.brain_file = brain_parm.brain_file;
	parm.brain_sphR = brain_parm.FS_right_sphere_file;
	parm.brain_sphL = brain_parm.FS_left_sphere_file;
	parm.key        = brain_parm.FS_sphere_key;
	vb_job_brain_add_sphcoord(proj_root, parm);

%--------------------------------------------------------------------------
% Make two area files derived from two anatomical divisons
%--------------------------------------------------------------------------
vb_atlas_label(brain_file, 'aal');
vb_remove_island_atlas(brain_file, 'aal');

%vb_atlas_label(brain_file, 'brodmann');
%vb_remove_island_atlas(brain_file, 'brodmann');

if DEBUG_MODE==1, return; end;

%----------------------------------------------
% Search neighbor points along cortex sheet
%----------------------------------------------

[nextIX , nextDD] = vb_find_neighbor_all(Rmax, xxF, xxD, Vindx, Display);

vb_disp('Save neighbor index of cortical vertex', ...
        VERBOSE_LEVEL_NOTICE);
vb_save([brain_file],'nextIX','nextDD');

%--------------------------------------
% Make new area and act files
%--------------------------------------
act_new.key     = 'Uniform';
act_new.xxP     = ones(Ndipole,1);
act_new.comment = 'homogeneous fMRI constraint';

vb_add_act([act_file], act_new, MRI_ID, OFF);

% Make Area file
Vindx  = [Vinfo.cortexL; Vinfo.cortexR];
AreaNew.key      = 'Cortex';
AreaNew.Iextract = Vindx; %[1:Ndipole]'; 

vb_add_area([area_file], AreaNew, MRI_ID, OFF);

%-----------------------------------------
% Check brain model by 3D & MRI image
%-----------------------------------------
vb_check_brain_model(proj_root,brain_parm);


%--------------------------
% Import inflate model
%--------------------------
if has_inflate_model(brain_parm)
  vb_job_inflate(proj_root, brain_parm);
end

%--------------------------
% Save project file
%--------------------------
proj_file = get_project_filename;
if ~isempty(proj_file)
    project_file_mgr('load', proj_file);
    project_file_mgr('add', 'brain_parm', brain_parm);
end

return;

function [result] = has_inflate_model(brain_parm)
% Check if brain_parm contains valid fields for inflate model.
% result : return bit pattern
%          = 01(1 as int): BV inflate data is set.
%          = 10(2 as int): FS inflate data is set.
%          = 11(3 as int): Both inflate model were set
result = 0;
BV_BIT = 1; FS_BIT = 2;
% bit all on
result = bitset(result, BV_BIT, 1);
result = bitset(result, FS_BIT, 1);

BV_fields = {'BV_left_infl_file', 'BV_right_infl_file'};
FS_fields = {'FS_left_infl_file',  'FS_right_infl_file', ...
             'FS_left_curv_file', 'FS_right_curv_file'};
for k=1:length(BV_fields)
  if ~isfield(brain_parm, BV_fields{k}) || ...
     exist(brain_parm.(BV_fields{k}), 'file') ~= 2
    % bit off 
    result = bitset(result, BV_BIT, 0);
    break;
  end
end
for k=1:length(FS_fields)
  if ~isfield(brain_parm, FS_fields{k}) || ...
     exist(brain_parm.(FS_fields{k}), 'file') ~= 2
    % bit off 
    result = bitset(result, FS_BIT, 0);
    break;
  end
end

function	[Vmni, origin] = convert_spm_mni(Vspm, fname)
% Transform SPM-Right-m coordinate to MNI mm coordinate 
%   [Vmni, origin] = convert_spm_mni(Vspm, fname)
% Vmni  ; MNI [mm] coordinate (Npoint x 3)
% fname : file name (MNI_152) of T1 image file
% Vspm  : SPM-Right [m] coordinate 
% origin: origin of MNI cordinate in SPM-Right [m] coordinate 
% Vspm  = Vmni + origin;  (Vmni = 0 at origin)

[Trans , dim] = get_coord_trans_mat(fname);

% Trans : transform matrix from voxcel to MNI mm coordinate
%   [x  y  z  1] = [i  j  k  1] * Trans
% Xmni = Vox  * Trans
% Vox  = Xmni * inv(Trans)

% Center position of 3D image in MNI mm
center = vb_affine_trans(dim/2 , Trans);

% Xspm = 0       <-> Xmni = center
% Xspm = -center <-> Xmni = 0
% Xspm = Xmni - center 
% Xmni = Xspm + center

% Get MNI mm coordinate
Vmni = vb_repadd(Vspm * 1000 , center) ;

% origin of MNI cordinate in SPM-Right coordinate 
%origin = -center ;
origin = -center * 0.001;
