function [ ch_type ] = neuromag_get_channel_type(type_key)
% return neuromag channel type information
% [usage]
%   [ ch_type ] = neuromag_get_channel_type( type_key )
% [input]
%   type_key : <optional> There are three patterns.
%               1) if this is empty or this is not be specified,
%                  return struct array of all channel types
%               2) if this is character, return corresponding type code number
%               3) if this is numeric, return corresponding type character
% [output]
%    ch_type : 1) <struct array> fields are as follows:
%               .id   : type code
%               .name : type name
%              2) <numeric> corresponding type code
%              3) <string> correstponding type name
% [note]
%   @see vb_yokogawa_get_channel_type : For yokogawa channel type
% [history]
%   2011-05-24 (Sako) initial version
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% --- CHECK ARGUMENTS --- %
if ~exist('type_key', 'var'), type_key = []; end
[type_key] = inner_check_arguments(type_key);

% --- MAIN PROCEDURE --------------------------------------------------------- %
%

% --- make an array of channel type definition

% --- DEFINITION of YOKOGAWA - as a baseline for definition of neuromag
type_index = 1;

NullChannel         = 0;
channel_type(type_index).id   = NullChannel;
channel_type(type_index).name = 'NullChannel';
type_index = type_index + 1;

MagnetoMeter        = 1;
channel_type(type_index).id   = MagnetoMeter;
channel_type(type_index).name = 'MagnetoMeter';
type_index = type_index + 1;

AxialGradioMeter    = 2;
channel_type(type_index).id   = AxialGradioMeter;
channel_type(type_index).name = 'AxialGradioMeter';
type_index = type_index + 1;

PlanarGradioMeter   = 3;
channel_type(type_index).id   = PlanarGradioMeter;
channel_type(type_index).name = 'PlanarGradioMeter';
type_index = type_index + 1;

% ReferenceChannelMark
RCMark = hex2dec('0100');

Ref_MagnetoMeter       = bitor(RCMark, MagnetoMeter);
channel_type(type_index).id   = Ref_MagnetoMeter;
channel_type(type_index).name = 'Ref_MagnetoMeter';
type_index = type_index + 1;

Ref_AxialGradioMeter   = bitor(RCMark, AxialGradioMeter);
channel_type(type_index).id   = Ref_AxialGradioMeter;
channel_type(type_index).name = 'Ref_AxialGradioMeter';
type_index = type_index + 1;

Ref_PlanarGradioMeter  = bitor(RCMark, PlanarGradioMeter);
channel_type(type_index).id   = Ref_PlanarGradioMeter;
channel_type(type_index).name = 'Ref_PlanarGradioMeter';
type_index = type_index + 1;

TriggerChannel      = -1;
channel_type(type_index).id   = TriggerChannel;
channel_type(type_index).name = 'TriggerChannel';
type_index = type_index + 1;

EegChannel          = -2;
channel_type(type_index).id   = EegChannel;
channel_type(type_index).name = 'EegChannel';
type_index = type_index + 1;

EcgChannel          = -3;
channel_type(type_index).id   = EcgChannel;
channel_type(type_index).name = 'EcgChannel';
type_index = type_index + 1;

EtcChannel          = -4;
channel_type(type_index).id   = EtcChannel;
channel_type(type_index).name = 'EtcChannel';
type_index = type_index + 1;


% ----- NEUROMAG
NEUROMAG_OFFSET = 0;
NEUROMAG_MagnetoMeter      = MagnetoMeter + NEUROMAG_OFFSET;      % 11
channel_type(type_index).id   = NEUROMAG_MagnetoMeter;
channel_type(type_index).name = 'NEUROMAG_MagnetoMeter';
type_index = type_index + 1;

NEUROMAG_PlanarGradioMeter = PlanarGradioMeter + NEUROMAG_OFFSET; % 13
channel_type(type_index).id   = NEUROMAG_PlanarGradioMeter;
channel_type(type_index).name = 'NEUROMAG_PlanarGradioMeter';
type_index = type_index + 1;

NEUROMAG_EegChannel = EegChannel - NEUROMAG_OFFSET;               % -12
channel_type(type_index).id   = NEUROMAG_EegChannel;
channel_type(type_index).name = 'NEUROMAG_EegChannel';
type_index = type_index + 1;

NEUROMAG_EtcChannel = EtcChannel - NEUROMAG_OFFSET;               % -14
channel_type(type_index).id   = NEUROMAG_EtcChannel;
channel_type(type_index).name = 'NEUROMAG_EtcChannel';
% type_index = type_index + 1;

if isempty(type_key)
  ch_type = channel_type;
  return;
elseif ischar(type_key)
  ch_type = inner_find_key_by_string(channel_type, type_key);
elseif isnumeric(type_key)
  ch_type = inner_find_key_by_numeric(channel_type, type_key);
end

return;
%
% --- END OF MAIN PROCEDURE -------------------------------------------------- %

% --- INNER FUNCTIONS -------------------------------------------------------- %
%
% --- inner_check_arguments()
%
function [type_key] = inner_check_arguments(type_key)
if isempty(type_key)
  % doing nothing means to return channel type array.
end
return;
%
% --- end of inner_check_arguments()

% --- inner_find_key_by_string()
%
function ch_type = inner_find_key_by_string(channel_type, type_key)
ch_type = [];

if ~ischar(type_key)
  warning('(%s) type_key is not character\n', 'inner_find_key_by_string');
  return;
end

n_type = length(channel_type);

for i_type = 1:n_type
  if strcmp(channel_type(i_type).name, type_key) == 1
    ch_type = channel_type(i_type).id;
    break;
  end
end
return;
%
% --- end of inner_find_key_by_string()

% --- inner_find_key_by_numeric()
%
function ch_type = inner_find_key_by_numeric(channel_type, type_key)
ch_type = [];

if ~isnumeric(type_key)
  warning('(%s) type_key is not numeric\n', 'inner_find_key_by_numeric');
  return;
end

n_type = length(channel_type);
for i_type = 1:n_type
  if channel_type(i_type).id == type_key
    ch_type = channel_type(i_type).name;
    break;
  end
end
return;
%
% --- end of inner_find_key_by_numeric()
%
% --- END OF INNER FUNCTIONS ------------------------------------------------- %

% --- END OF FILE --- %

