function [optlambda, optsigma2, optcriteria, Jest, F, Info] = wmn_solution(B, G, X, w, option)
% Weighted Minimum Norm solution for the static inverse problem.
%  
% Solve the following minimization problem  
%   min {E(J) = (B-GJ)'X(B-GJ) + lam*J'WJ}  
% with optimization of 'lam' by ABIC or GCV. 
% When 'option.lapacian' exists, the following minimization problem
%   min {E(J) = (B-GJ)'X(B-GJ) + lam*J'LWLJ}  
% is solved ('LORETA' solution).
%  
% [Syntax] 
%   > [lambda, sigma2, ABIC, Jest, F, Info] = wmn_solution(B, G, Ceps, W, option)
%
% [Example]
% >> option.criteria = 'ABIC'
% >> option.flag_current = 1;
% >> [lambda, sigma2, ABIC, Jest, F] = wmn_solution(B, G, Ceps, W, option)
%
% returns optimal parameters, estimates of J, and the inverse filter using ABIC 
% as the statisitcal criteria for optimization.
%
% [Output]
%   optlambda : optimum regularization parameter
%   optsigma2 : optimum observation variance
%   optcriteria : the value of criteria (ABIC or GCV) at optlambda
%   Jest : estimated J
%   F    : inverse filter (mapping from B to J) 
%   Info : information of the optimization 
%        1st column : lambda
%        2nd column : criteria (ABIC or GCV)
%        3rd column : observation noise variance (SIG2)
%
% [Input]
%   B : EEG/MEG observation of a single trial (Nch*Nt matrix)
%   G : lead field matrix  (Nch * Nj matrix)
%   X : observation weight matrix (inverse of observation noise)
%   w : the diagonal elements of prior weight matrix W (should be diagonal, because of memory problem) 
%
% [Optional Input]
%   option.criteria : criteria for the estimation of reguralization parameter 
%   option.laplacian : if exist, W is replaced by LWL (L is lapalcian)
%   option.flag_plot : if 1, the values of 'criteria' (ABIC or GCV) are displayed with respect to 'lambda' 
%   option.flag_current : if 1, the current is reconstrcted in the third output (default:1)  
%   option.Npoints : the number of grid points in lambda axis to be explored (default:100) 
%   option.lambda  : fix the regularization parameter  (default : []) 
%   option.search_margin : parameter to define range of lambda to be searched (default:5).
%
% 2016-03-22 O.Yamashita
% * allow empty X and w to speed up (avoid time-consuming sqrtm function)
% 2014-03-03 O.Yamashita 
% * add a new field 'search_margin' to 'option' struct
% 2014-02-10 O.Yamashita
% * add a new field 'lambda' to 'option' struct
% * modify comments 
% 2012-11-07 O.Yamashita
% * save information for the optimization in 'Info' 
% 2012-04-26 O.Yamashita
% * bug fix for the lines to compute square root of a matrix (sqrt --> sqrtm)
%   previous version causes complex values 
% * add inverse filter 'F' as an output argument (important)
% 2005-04-26 O.Yamashita beta-version, 

%profile on

[Nch, Nt] = size(B);
[Nj] = size(G,2);


% default 
criteria = 'ABIC';
flag_plot = 1;
flag_current = 1;
L = speye(Nj);
Npoints = 100;   % Number of points on the curve.
lambda = [];
search_margin = 5;  % in log scale

%% Error check 
if exist('option') == 1  & isfield(option, 'criteria')
    criteria = option.criteria;
end
if exist('option') == 1  & isfield(option, 'flag_plot')
    flag_plot = option.flag_plot ;
end
if exist('option') == 1  & isfield(option, 'flag_current')
    flag_current = option.flag_current;
end  
if exist('option') == 1  & isfield(option, 'laplacian')
    L = option.laplacian;
    L = L / max(L(:));
end
if exist('option') == 1  & isfield(option, 'Npoints')
    Npoints = option.Npoints;
end
if exist('option') == 1  & isfield(option, 'lambda')
    lambda = option.lambda;
end
if exist('option') == 1  & isfield(option, 'search_margin')
    search_margin = option.search_margin;
end

% pre-whitening 
if isempty(X)
    Xsqrt = eye(Nch);
else
    Xsqrt = sqrtm(X);
end
if isempty(w)
    wsqrt = ones(Nj,1);
else
    wsqrt = sqrt(w);
end

Btil = Xsqrt * B; 
Gtil = Xsqrt * G / L * spdiags(1./wsqrt, 0, Nj, Nj) ;

[U s V] = csvd1(Gtil);

% Sets grid points
if isempty(lambda)
    minloglx = log(s(end)) - search_margin;
    maxloglx = log(s(1)) + search_margin;
    bin = (maxloglx-minloglx)/(Npoints-1);
    loglx_points = [minloglx : bin : maxloglx];
else
    loglx_points = log(lambda);   % fixed lambda
end

Npoints = length(loglx_points); 
SIG2 = zeros(Npoints,1);
CRITERIA = zeros(Npoints,1);

UtB = U'*Btil;
BB = UtB * UtB';
s2=s.*s; 

% find optimum regularization parameter
switch criteria
    case 'ABIC'
        %% calculate ABIC
        for i = 1 : Npoints
          lam = exp(loglx_points(i));         
          l2=lam;
          r = l2./(s2+l2); 
          sig2 = sum(sum(diag(r).*BB))/Nch/Nt; %tr(diag(r)*BB)
          ABIC1=Nch*Nt*log(sig2);
          ABIC2= -Nt*sum(log(r));
          CRITERIA(i)=ABIC1+ABIC2;
          SIG2(i) = sig2;
        end
        
    case 'GCV'
        %% calculate GCV
        for i = 1 : Npoints
          lam = exp(loglx_points(i));         
          l2=lam;
          r = l2./(s2+l2);
          nume = sum(sum(diag(r.^2).*BB));
          denom = (sum(r))^2;
          GCV = nume/denom;
          sig2 = sum(sum(diag(r).*BB))/Nch/Nt; %tr(diag(r)*BB)
          CRITERIA(i) = GCV;
          SIG2(i) = sig2;
        end
        
    otherwise
        disp('No such a method');
end

[tmp, ind] = min(CRITERIA);
optlambda = exp(loglx_points(ind));
optsigma2 = SIG2(ind);
optcriteria = CRITERIA(ind);
Info = [exp(loglx_points(:)) CRITERIA(:) SIG2(:)];

%% current reconstruction

if flag_current 
    l2= optlambda;
    r = s./(s2+l2); 
    Jtil = V *diag(r)*UtB;
    Jest = L\ Jtil ./ wsqrt(:,ones(1,Nt));  %Jest = L\ spdiags(1./wsqrt, 0, Nj, Nj) * Jtil;
    F = L\spdiags(1./wsqrt, 0, Nj, Nj)*V*diag(r)*U'*Xsqrt;
else
    Jest = [];
    F = [];
end

if flag_plot
    figure
    [AX,H1,H2] = plotyy(loglx_points, CRITERIA, loglx_points, SIG2);
    xlabel('Log of lambda');
    set(get(AX(1),'Ylabel'),'String','ABIC or GCV')
    set(get(AX(2),'Ylabel'),'String','Observation Noise Variance');
    hold on;
    plot([loglx_points(ind) loglx_points(ind)], [min(get(AX(1),'ytick')) max(get(AX(1),'ytick'))], 'k--');
end
    

%profile viewer


