function [Jpred] = forward_prediction(MAR, Delta, omega)
% Compute current sources predicted by a linear MAR model
%
% Input
%     MAR: MAR matrix                 [Nvact x Nvact]
%   Delta: Time lag matrix            [Nvact x Nvact]
%   omega: Input signals to the model [Nvact x Nt]
%
% Output
%   Jpred: Predicted current sources  [Nvact x Nt]
%
% 2014/10/20 M.Fukushima
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)

% # of sources
Nvact = size(MAR,1);
% # of time instances
Nt = size(omega,2);
% Binarized structural connectivity matrix
Ind = (Delta~=0);

% Connection numbers
sum_ind = sum(Ind,2);
% Time lags
unidlt = full(setdiff(unique(Delta),0));

% Store matrix entries into a smaller size of matrix
Indx   = zeros(Nvact,max(sum_ind));
Deltax = zeros(Nvact,max(sum_ind));
MARx   = zeros(Nvact,max(sum_ind));
for nv = 1:Nvact
  Indx(nv,1:sum_ind(nv))   = find(Ind(nv,:)==1);
  Deltax(nv,1:sum_ind(nv)) = Delta(nv,Delta(nv,:)~=0);
  MARx(nv,1:sum_ind(nv))   = MAR(nv,Ind(nv,:)~=0);
end

% Forward prediction by a linear MAR model
Jpred = zeros(Nvact, Nt);
for t = 1:Nt
  Jtmp = [zeros(Nvact, unidlt(end)) Jpred];
  t_end = t + unidlt(end);
  
  % Prediction [MEX ver.]
  AJpred = atimesj(MARx, Indx, Deltax, sum_ind, ...
    Jtmp(:,t_end:-1:(t_end-unidlt(end))));
%   % Prediction [Matlab ver.]
%   AJpred = zeros(Nvact,1);
%   for nv = 1:Nvact
%     for ii = 1:sum_ind(nv)
%       AJpred(nv) = AJpred(nv) + ...
%         MARx(nv,ii) * Jtmp(Indx(nv,ii),t_end-Deltax(nv,ii));
%     end
%   end

  % Predicted current sources
  Jpred(:,t) = AJpred + omega(:,t);
end
