function rgb_val = rgb_val_get(color_name)
% get rgv value from color name.
%
% [Usage]
%    rgb_val = rgb_val_get(color_name)
%
% [Input]
%    color_name : color name/namelist      [string/cell string]
%    These short/long name can be used as color name.
%       'y'  : 'yellow'
%       'm'  : 'magenta'
%       'c'  : 'cyan'
%       'r'  : 'red'
%       'g'  : 'green'
%       'b'  : 'blue'
%       'w'  : 'white'
%       'k'  : 'black'
%       'o'  : 'orange'
%       'db' : 'darkbrown'
%       'nb' : 'navyblue'
%       'sb' : 'skyblue'
%       'lb' : 'lightblue'
%       'br' : 'brown'
%       'yg' : 'yellowgreen'
%       'bg' : 'bottlegreen'
%
% [Output]
%       rgb_val : rgb value (R G B)  [Nx3]
%                 The range of R/G/B is from 0 to 1.
%
% [Example]
%    % get orange value
%    rgb_val = rgb_val_get('o');
%    rgb_val = rgb_val_get('orange');
%
%    % get multiple rgb value
%    rgb_vals = rgb_val_get({'y', 'm', 'c'});
%
% Copyright (C) 2011, ATR All Rights Reserved.
% License : New BSD License(see VBMEG_LICENSE.txt)


% RGB value                 % color name
rgb_table = {
    {[1 1 0],               {'y',   'yellow'}};
    {[1 0 1],               {'m',   'magenta'}};
    {[0 1 1],               {'c',   'cyan'}};
    {[1 0 0],               {'r',   'red'}};
    {[0 1 0],               {'g',   'green'}};
    {[0 0 1],               {'b',   'blue'}};
    {[1 1 1],               {'w',   'white'}};
    {[0     0         0],   {'k',   'black'}};
    {[1     0.648     0],   {'o',   'orange'}};
    {[0.435 0.294 0.243],   {'db',  'darkbrown'}};
    {[0     0       0.5],   {'nb',  'navyblue'}};
    {[0.529 0.808 0.922],   {'sb',  'skyblue'}};
    {[0.4   1         1],   {'lb',  'lightblue'}};
    {[0.647 0.165 0.165],   {'br',  'brown'}};
    {[0.604 0.804 0.196],   {'yg',  'yellowgreen'}};
    {[0.2   0.377 0.271],   {'bg',  'bottlegreen'}};
};

%
% --- Previous check
%
if nargin ~= 1
    help(mfilename);
    return;
end

%
% --- Main Procedure
%
if ischar(color_name)
    color_name = {color_name};
end
Ncolors = length(color_name);

rgb_val = [];
for k=1:Ncolors
    rgb_val = [rgb_val; inner_find_color(rgb_table, color_name{k})];
end

function [rgb_value] = inner_find_color(rgb_table, color_name)
% find rgb value by searching color table.
%
% [Input]
%    color_name : string
% [Output]
%     rgb_value : [R G B] or empty(when not found).

rgb_value = [];

for k=1:length(rgb_table)
    color_name_def = rgb_table{k}{2};
    ix = strmatch(lower(color_name), lower(color_name_def), 'exact');
    if ~isempty(ix)
        rgb_value = rgb_table{k}{1};
        break;
    end
end
if isempty(rgb_value)
    warning('The specified color : %s not found.', color_name);
end
